-- Belegarten
CREATE TABLE belart (
    ba_id               SERIAL PRIMARY KEY,
    ba_bez              VARCHAR(50)
 );

 --

 CREATE OR REPLACE FUNCTION belart__a_iud() RETURNS TRIGGER AS $$
  BEGIN
    IF NOT tg_op='INSERT' THEN
        DELETE FROM belartlang WHERE bal_ba_id=old.ba_id AND bal_spr_key=prodat_languages.current_lang();
    END IF;
    IF NOT tg_op='DELETE' THEN
        INSERT INTO belartlang (bal_ba_id, bal_spr_key, bal_txt) VALUES (new.ba_id, prodat_languages.current_lang(), new.ba_bez);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belart__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belart
    FOR EACH ROW
    EXECUTE PROCEDURE belart__a_iud();
--

-- Übersetzungen für Belegarten
CREATE TABLE belartlang (
    bal_id              SERIAL PRIMARY KEY,
    bal_ba_id           INTEGER NOT NULL REFERENCES belart ON UPDATE CASCADE ON DELETE CASCADE,
    bal_spr_key         VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
    bal_txt             VARCHAR(50)
 );

 CREATE UNIQUE INDEX belartlang_unique ON belartlang(bal_ba_id, bal_spr_key);

--

-- Erlöskonten (Buchhaltung)
CREATE TABLE erloes (
    er_kto              VARCHAR(6) PRIMARY KEY,
    er_bez              VARCHAR(75),
    er_automatik        BOOL NOT NULL DEFAULT TRUE,
    er_erlkto           BOOL, -- Erloeskonto, Dobitorenkonto
    er_aufkto           BOOL  -- Aufwandskonto, Kreditorenkonto
    --er_debi             BOOL, -- Debitorenkonto, gelöscht weil sinnlos (07.06.2017; JM-MF-DS) = er_aufkto, #2829
    --er_kred             BOOL  -- Kreditorenkonto, gelöscht weil sinnlos (07.06.2017; JM-MF-DS) = er_erlkto, #2829
);

CREATE OR REPLACE FUNCTION erloes__b_100_iu__xxxkto() RETURNS TRIGGER AS $$
  -- Wenn Aufwandskonto gesetzt wird und Erloes NULL ist => false setzen sowie vice versa
  BEGIN
    IF new.er_erlkto AND new.er_aufkto IS NULL THEN
        new.er_aufkto := false;
    END IF;

    IF new.er_aufkto AND new.er_erlkto IS NULL THEN
        new.er_erlkto := false;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER erloes__b_100_iu__xxxkto
    BEFORE INSERT OR UPDATE
    OF er_erlkto, er_aufkto
    ON erloes
    FOR EACH ROW
    EXECUTE PROCEDURE erloes__b_100_iu__xxxkto();
--

-- #8906 Erlöskonten zusammenFassung (Buchhaltung)
CREATE TABLE erloes_zf (
    ezf_id              SERIAL PRIMARY KEY,                  -- ID
    ezf_steu_z          INTEGER NOT NULL REFERENCES steutxt ON UPDATE CASCADE,    -- Steuercodes
    ezf_sachkonto       VARCHAR(6) NOT NULL REFERENCES erloes ON UPDATE CASCADE,  -- Sachkonto (AC / Artikelgruppe)
    ezf_buchkonto       VARCHAR(6) REFERENCES erloes ON UPDATE CASCADE,  -- Erlöskonten (Controlling/Kostenrechnung)
    ezf_steukonto       VARCHAR(6) REFERENCES erloes ON UPDATE CASCADE              -- Steuerliches Konto (Umsatz Inland, Umsatz Ausland)
 );

 CREATE UNIQUE INDEX  xtt29235 ON erloes_zf (ezf_steu_z, ezf_sachkonto, ezf_buchkonto);  -- Kontenvariante darf nur ein mal sein


-- Faktura, Ausgangsrechnungen (Kopf)
CREATE TABLE belkopf (
    be_bnr              VARCHAR(15) NOT NULL CONSTRAINT xtt2114 PRIMARY KEY,
    be_prof             VARCHAR(1),                             -- R=Rechnung, G=Gutschrift, P=Proforma, T=Teilzahlung; und bei 'T'/'R' noch be_txba beachten
    be_rkrz             VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --Rechnungsadresse
    be_spco             INTEGER NOT NULL REFERENCES adkspco,
    be_skv              SMALLINT,
    be_sks              NUMERIC(8,2),
    be_zak              SMALLINT NOT NULL DEFAULT 3,
    be_ms               SMALLINT,
    be_mahnsperr        BOOL NOT NULL DEFAULT FALSE,
    be_mahnd            DATE,
    be_rund             NUMERIC(8,2),
      be_umf              NUMERIC(8,2) DEFAULT 1,
    be_umr              NUMERIC(8,2) NOT NULL DEFAULT 1,             --Umrechnungsfaktor (Kurs)
    be_bdat             DATE DEFAULT current_date,              --Erstelldatum Beleg?
    be_valuta           DATE DEFAULT current_date,
    be_waco             VARCHAR(3) NOT NULL CONSTRAINT xtt4056__be_waco REFERENCES bewa, --Rechnungswährung
    be_wabk             VARCHAR(5),
    be_kond             VARCHAR(100),
    be_txba             INTEGER NOT NULL REFERENCES belart,--Belegart
    be_abprozent        NUMERIC(5,2),--Abschlagszahlung Prozent
      be_steucode1        INTEGER REFERENCES steutxt,
      be_steuproz1        NUMERIC(8,4),
      be_steucode2        INTEGER REFERENCES steutxt,
      be_steuproz2        NUMERIC(8,4),
    be_bem1             TEXT,
    be_bem1_rtf         TEXT,
    be_bem2             TEXT,
    be_bem2_rtf         TEXT,
    be_txtexp           TEXT,
    be_txtexp_rtf       TEXT,
    be_txtexpurspr      TEXT,
    be_txtexpurspr_rtf  TEXT,
    be_sta              SMALLINT,
    be_gew              NUMERIC(12,2),
    be_vers             VARCHAR(75),                            -- Versandart
    be_reeb             NUMERIC,                                -- Rechnungseinbehalt
    be_rebd             DATE,                                   -- Rechnungseinbehalts-Datum (bis) (5% bis nächstes Jahr)
    be_zahl_erledigt    BOOL NOT NULL DEFAULT FALSE,
    be_def              BOOL NOT NULL DEFAULT FALSE,            -- Definitivstatus Belege
    be_edidat           DATE,                                   -- EDI-Exportdatum via APPS, Rechnungsexport an Kunde
    be_buchdat          DATE,                                   -- Beleg ist an die Buchhaltungsschnittstelle übergeben
    be_buchmonth        VARCHAR,                                -- Buchungsmonat für Buchhaltung
    --be_fixwert          BOOLEAN NOT NULL DEFAULT FALSE,
    be_gesamt_net           NUMERIC(16,4) NOT NULL DEFAULT 0,       -- Belegsumme
    be_gesamt_net_basis_w   NUMERIC(16,4) DEFAULT 0,
    be_gesamt_steu          NUMERIC(16,4) NOT NULL DEFAULT 0,       -- Belegsumme nach Steuern
    be_gesamt_steu_basis_w  NUMERIC(16,4) DEFAULT 0,
    be_invckn_edi       INTEGER,
    --be_esrpz1           SMALLINT, --entfernt #11311 18.08
    --be_esrpz2           SMALLINT, --entfernt #11311 18.08
    be_apkrzl           VARCHAR(10),
    be_ap               VARCHAR(100),                            -- Freies Textfeld, Ansprechpartner
    be_allg1            VARCHAR(100),
    be_allg2            VARCHAR(100),
    be_allg3            VARCHAR(100),
    be_allnr1           NUMERIC(12,2),
    be_allnr2           NUMERIC(12,2),
    be_allnr3           NUMERIC(12,2),
    be_zahlart          SMALLINT REFERENCES zahlart ON UPDATE CASCADE,
    be_titel            VARCHAR(175), -- Titel der Rechnung
    be_liefkrz          VARCHAR(30) REFERENCES adressen_keys ON UPDATE CASCADE, -- Lieferadresse
    be_sortkrz          VARCHAR(30) REFERENCES adressen_keys ON UPDATE CASCADE, -- Standort
    be_apint            VARCHAR(10) DEFAULT tsystem.current_user_ll_db_usename(), -- CONSTRAINT llv Ansprechpartner Intern...Standard angemeldeter Nutzer
    be_ktv_name         VARCHAR(100) REFERENCES ktovz ON UPDATE CASCADE,
    be_pruefer          VARCHAR,                                -- Mitarbeiternummer des zuständigen Mitarbeiters für Prüfung
    be_freigabe         BOOLEAN NOT NULL DEFAULT FALSE          -- Chef hat Zahlung freigegeben
 );

 -- Indizes
    CREATE INDEX belkopf_be_rkrz ON belkopf (be_rkrz, be_def);
    CREATE INDEX belkopf_be_liefkrz ON belkopf(be_liefkrz);
    CREATE INDEX belkopf_be_sortkrz ON belkopf(be_sortkrz);

    CREATE INDEX belkopf_be_bnr_like ON belkopf (be_bnr varchar_pattern_ops);
    CREATE INDEX belkopf_be_rkrz_like ON belkopf (be_rkrz varchar_pattern_ops);

    CREATE INDEX belkopf_be_bdat ON belkopf (be_bdat, be_def);
    CREATE INDEX belkopf_ym_bdat ON belkopf (date_to_yearmonth(be_bdat));
 --

 -- Sequenzen
    --CREATE SEQUENCE belkopf_gut_be_bnr_seq;
    CREATE SEQUENCE belkopf_prof_be_bnr_seq;

    CREATE SEQUENCE belkopf_invckn; --für EDI-Export einfach eine fortlaufende Nummer
 --

 --
 CREATE OR REPLACE FUNCTION belkopf__b_i() RETURNS TRIGGER AS $$
  DECLARE krz VARCHAR;
          export BOOLEAN;
          euexport BOOLEAN;
          ursprungtxt BOOLEAN;
          adktxtexport TEXT;
          adktxtexportrtf TEXT;
          adktxtursprung TEXT;
          adktxtursprungrtf TEXT;
  BEGIN
    SELECT ad_krz, a1_export, a1_euexport, a1_ursprungstxt,
           a1_export_adktxt, a1_export_adktxt_rtf,
           a1_ursprungstxt_adktxt, a1_ursprungstxt_adktxt_rtf
    INTO krz, export, euexport, ursprungtxt,
         adktxtexport, adktxtexportrtf,
         adktxtursprung, adktxtursprungrtf
    FROM adressen_view JOIN adk1 ON a1_krz=adk_ad_krz WHERE ad_krz=new.be_rkrz;
    --
    IF new.be_bem1 IS NULL THEN
        SELECT txt, txtrtf INTO new.be_bem1, new.be_bem1_rtf FROM belarzu__zu_tit__gettxtrtf('DOKFAKT_TXT_KOPF', prodat_languages.adk1_spco(krz));
    END IF;
    IF new.be_bem2 IS NULL THEN
        SELECT txt, txtrtf INTO new.be_bem2, new.be_bem2_rtf FROM belarzu__zu_tit__gettxtrtf('DOKFAKT_TXT_FUSS', prodat_languages.adk1_spco(krz));
    END IF;
    --
    IF new.be_txtexp IS NULL THEN
        IF adktxtexport IS NOT NULL THEN --Kundenspezifischer Exporttxt
            new.be_txtexp:=adktxtexport;
            new.be_txtexp_rtf:=adktxtexportrtf;
        ELSIF export THEN                --StandardExportTxt aus Bemerkungsverwaltung
            SELECT txt, txtrtf INTO new.be_txtexp, new.be_txtexp_rtf FROM belarzu__zu_tit__gettxtrtf('EXPORT-INT', prodat_languages.adk1_spco(krz));
        ELSIF euexport THEN              --
            SELECT txt, txtrtf INTO new.be_txtexp, new.be_txtexp_rtf FROM belarzu__zu_tit__gettxtrtf('EXPORT-EU', prodat_languages.adk1_spco(krz));
        END IF;
    END IF;
    --
    IF new.be_txtexpurspr IS NULL THEN
        IF adktxtursprung IS NOT NULL THEN
            new.be_txtexpurspr:=adktxtursprung;
            new.be_txtexpurspr_rtf:=adktxtursprungrtf;
        ELSIF ursprungtxt THEN--noch kein Text vorhanden (Proformarechnung)
            SELECT txt, txtrtf INTO new.be_txtexpurspr, new.be_txtexpurspr_rtf FROM belarzu__zu_tit__gettxtrtf('EXPORT-URSPRUNG', prodat_languages.adk1_spco(krz));
        END IF;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belkopf__b_i
    BEFORE INSERT
    ON belkopf
    FOR EACH ROW
    EXECUTE PROCEDURE belkopf__b_i();
 --

 --
 CREATE OR REPLACE FUNCTION belkopf__b_iu() RETURNS TRIGGER AS $$
  DECLARE belzeil_fehlerhaft VARCHAR;
  BEGIN
    -- Fehlerbehandlungen
        -- Abschlagsprozente in nicht berücksichtigten Rechnungsarten
        IF new.be_abprozent IS NOT NULL AND new.be_prof NOT IN ('T', 'P') THEN
            RAISE EXCEPTION '%', lang_text(11089);--Abschlagsprozente sind nur in Teilzahlungen oder Proforma möglich!
            new.be_abprozent:=NULL;--?? oder lieber so?
        END IF;
        /* wird jetzt richtig berücksichtig: wenn Preis enthält Unterposition, aber keine Unterposition, ist Preis dennoch gültig. DS 2018-06-02
        -- Prüfungen, wenn Rechnung definitiv gesetzt wird
        IF TG_OP = 'UPDATE' THEN
            IF new.be_def AND NOT old.be_def THEN
                -- Positionsoption 'Preis enthält Unterpositionen' ist aktiv, ohne dass Unterpositionen mit Bezug zu der zugehörigen Auftragsposition vorhanden sind.
                belzeil_fehlerhaft:= -- fehlerhafte Rechnungspositionen sammeln, wird zur Prüfung und in Fehlermeldung verwendet.
                    (SELECT string_agg(hpos.bz_pos, ', ' ORDER BY hpos.bz_pos) FROM belzeil_grund AS hpos
                     WHERE hpos.bz_be_bnr = new.be_bnr
                       AND hpos.bz_vkptotalpos
                       -- Hauptposition nicht angg. oder existiert nicht, vgl. belzeil_do_auftgsubpos, belzeil_auftg_lif__a_10_iud
                       AND NOT EXISTS(SELECT true FROM belzeil_grund AS upos WHERE upos.bz_be_bnr = hpos.bz_be_bnr AND upos.bz_auftg = hpos.bz_auftg AND upos.bz_auftghpos = hpos.bz_auftgpos)
                    );
                IF nullif(belzeil_fehlerhaft, '') IS NOT NULL THEN
                    RAISE EXCEPTION '%', lang_text(16583) || E'\n\n' || lang_text(6162) || ': ' || new.be_prof || ' ' || new.be_bnr || E'\n' || lang_text(633) || ': ' || belzeil_fehlerhaft;
                END IF;
                --
            END IF;
        END IF;
        */
    --

    -- Belegsummen
    new.be_gesamt_net_basis_w:=  new.be_gesamt_net  * new.be_umr; --das wird von Belzeils berechnet
    new.be_gesamt_steu_basis_w:= new.be_gesamt_steu * new.be_umr;
    --

    -- Steuern unbekannt, wenn kein Steuercode angg. ist.
    IF new.be_steucode1 IS NULL THEN
        new.be_steuproz1:= NULL;
    END IF;

    IF new.be_steucode2 IS NULL THEN
        new.be_steuproz2:= NULL;
    END IF;
    --

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belkopf__b_iu
    BEFORE INSERT OR UPDATE
    ON belkopf
    FOR EACH ROW
    EXECUTE PROCEDURE belkopf__b_iu();
 --

 --
 CREATE OR REPLACE FUNCTION belkopf__a_iu() RETURNS TRIGGER AS $$
  DECLARE doabschlagcalc BOOLEAN;
  BEGIN
    IF new.be_prof IN ('T', 'P') AND tg_op='UPDATE' THEN
        doabschlagcalc:=COALESCE(new.be_abprozent,-1)<>COALESCE(old.be_abprozent,-1);
    END IF;
    --
    IF doabschlagcalc /* AND NOT be_fixwert*/THEN
        UPDATE belkopf SET be_gesamt_net=COALESCE(bel_gesamt_betrag(be_bnr), 0), be_gesamt_steu=COALESCE(bel_gesamt_steu(be_bnr), 0) WHERE be_bnr=new.be_bnr;
    END IF;

    --Meldung ausgeben,wennn Steuersatz ungültig
    IF (new.be_steucode1 = (SELECT steu_z FROM steutxt WHERE steu_z=new.be_steucode1 AND steu_valid_to<current_date)) OR
       (new.be_steucode2 = (SELECT steu_z FROM steutxt WHERE steu_z=new.be_steucode2 AND steu_valid_to<current_date))
    THEN
        PERFORM PRODAT_TEXT(15810);
    END IF;

    -- Gutschrift / Rechnungskorektur https://redmine.prodat-sql.de/issues/22530
    IF     (tg_op = 'UPDATE' )
       AND (new.be_prof = 'G' AND new.be_txba IS DISTINCT FROM old.be_txba)
    THEN
        UPDATE belzeil_grund SET bz_id = bz_id WHERE bz_be_bnr = new.be_bnr;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belkopf__a_iu
    AFTER INSERT OR UPDATE
    ON belkopf
    FOR EACH ROW
    EXECUTE PROCEDURE belkopf__a_iu();
 --

 --
 CREATE OR REPLACE FUNCTION belkopf__a_u() RETURNS TRIGGER AS $$
  BEGIN
    IF current_user='syncro' THEN
        RETURN new;
    END IF;
    IF COALESCE(new.be_gesamt_net,0) <> COALESCE(old.be_gesamt_net,0) OR
       COALESCE(new.be_gesamt_net_basis_w,0) <> COALESCE(old.be_gesamt_net_basis_w,0) OR
       COALESCE(new.be_gesamt_steu,0) <> COALESCE(old.be_gesamt_steu,0) OR
       COALESCE(new.be_gesamt_steu_basis_w,0) <> COALESCE(old.be_gesamt_steu_basis_w,0)
    THEN
        RETURN new;
    END IF;--es wird nur der Gesamtbetrag nachgetragen
    --Änderung Steuersatz, in Belegzeilen nachziehen
    PERFORM disablemodified();

    -- #22326 Steuercode auch dann setzen, wenn sich nur dieser geändert hat
    IF ( new.be_steuproz1 IS DISTINCT FROM old.be_steuproz1 ) OR ( old.be_steucode1 IS DISTINCT FROM new.be_steucode1 ) THEN
        UPDATE belzeil_grund SET bz_steucode = new.be_steucode1, bz_steuproz = new.be_steuproz1
        WHERE bz_be_bnr = new.be_bnr AND bz_steucode = old.be_steucode1;
    END IF;
    IF ( new.be_steuproz2 IS DISTINCT FROM old.be_steuproz2 ) OR ( old.be_steucode2 IS DISTINCT FROM new.be_steucode2 )THEN
        UPDATE belzeil_grund SET bz_steucode = new.be_steucode2, bz_steuproz = new.be_steuproz2
        WHERE bz_be_bnr = new.be_bnr AND bz_steucode = old.be_steucode2;
    END IF;

    IF (new.be_def AND (NOT old.be_def)) AND TSystem.Settings__GetBool('LieferscheinWADatumPerDefinitivRechnung') THEN -- Warenausgangsdatum vom Lieferschein spätestens dann setzen, wenn Rechnung definitiv wird. Oder nur manuell.
        PERFORM disableBelPosTrigger();
        UPDATE belegpos SET belp_termin = current_date
        FROM  belzeil_grund
        WHERE belp_termin IS NULL
          AND belp_belegtyp = 'LFS'
          AND bz_belp_id=belp_id
          AND bz_be_bnr=new.be_bnr;
        PERFORM EnableBelPosTrigger();
    END IF;

    PERFORM enablemodified();
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belkopf__a_u
    AFTER UPDATE
    ON belkopf
    FOR EACH ROW
    EXECUTE PROCEDURE belkopf__a_u();
 --

 --
 CREATE OR REPLACE FUNCTION belkopf__a_iu_keywordsearch() RETURNS TRIGGER AS $$
  BEGIN
   PERFORM TSystem.kws_create_keywords(new.*);
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belkopf__a_iu_keywordsearch
   AFTER INSERT OR UPDATE
    OF be_bnr
   ON belkopf
   FOR EACH ROW
   EXECUTE PROCEDURE belkopf__a_iu_keywordsearch();
 --

--
CREATE OR REPLACE FUNCTION belkopf__a_d() RETURNS TRIGGER AS $$
  BEGIN
    -- #9555 PHKO löschen von Eintrag im Zahlungsplan
    UPDATE auftgdokzahlplan SET azp_be_bnr = NULL WHERE azp_be_bnr = old.be_bnr;

    -- ggf reservierte Nummer zurücksetzen https://redmine.prodat-sql.de/issues/16290
    PERFORM num_unreserve(ident.fields, old.be_bnr)
            -- man könnte mittels Prüfung auf be_prof gleich die richtige Nummer (ident.fields) zurücksetzen
       FROM (SELECT unnest(ARRAY['BE_BNR', 'BE_BNR_P', 'BE_BNR_CASH', 'BE_BNR_G', 'BE_BNR_A'])::varchar AS fields) as ident;

     RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belkopf__a_d
    AFTER DELETE
    ON belkopf
    FOR EACH ROW
    EXECUTE PROCEDURE belkopf__a_d();
--

--- #22955
CREATE OR REPLACE FUNCTION belkopf__b_d() RETURNS TRIGGER AS $$
 BEGIN

    raise exception '%', format( lang_text( 39033 ), old.be_bnr ); --- Die Rechnung "%s"darf nicht gelöscht werden, weil die definitiv ist

    RETURN old;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER belkopf__b_d
 BEFORE DELETE
 ON belkopf
 FOR EACH ROW
 WHEN ( old.be_def )
 EXECUTE PROCEDURE belkopf__b_d();

-- Abzüge/Zuschläge
CREATE TABLE belabzu (                                                               -- VIEWFELDER
  beaz_id                SERIAL PRIMARY KEY,                                           -- ID des Zuschlags
  beaz_type              VARCHAR(1) DEFAULT 'E', -- Abzuschlagstyp                     -- Zuschlagstyp, E-Einmalig, P-Position, M-Per ME
  beaz_pos               INTEGER,                                                      -- Position
  beaz_abz_id            INTEGER NOT NULL REFERENCES abzu,                             -- ID der Vorgabe
  beaz_bebnr             VARCHAR(15) NOT NULL REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE,    -- ID des Datensatzes an dem der Zuschlag hängt
  beaz_anz               NUMERIC(12,4) NOT NULL DEFAULT 1,                             -- Anzahl / Menge
  beaz_abzubetrag        NUMERIC(12,4) NOT NULL DEFAULT 0,                             -- Betrag in Währung des Parent-Datensatzes
  beaz_abzuproz          NUMERIC(12,4),                                                -- Prozentualer Zuschlag auf Basis des Parent-Betrags
  beaz_canSkonto         BOOLEAN NOT NULL DEFAULT TRUE,                                -- Gilt Skonto für den Zuschlag?
  beaz_steu              INTEGER REFERENCES steutxt,                                   -- Steuercode (meist gleich Parent)
  beaz_steuproz          NUMERIC(5,2) NOT NULL DEFAULT 0,                              -- Prozentsatz der Steuer
  beaz_konto             VARCHAR(25),                                                  -- Kontierung
  beaz_visible           BOOLEAN NOT NULL DEFAULT TRUE,                                -- Auf Dokument sichtbar oder nicht?
  beaz_zutxt             TEXT,                                                         -- Zusätzlicher Hinweistext
  beaz_zutxt_rtf         TEXT,                                                         -- Zusätzlicher Hinweistext (RTF)
  beaz_zutxt_int         TEXT,                                                         -- Interner zusatztext (erscheint nicht auf Dokumenten)
  beaz_source_table      VARCHAR(40),                                                  -- Aus welcher Quelle wurde Abzu hierhin kopiert
  beaz_source_dbrid      VARCHAR(32),                                                  -- Aus welchem Datensatz wurde Abzu hierhin kopiert
  -- ZUSATZFELDER
  beaz_agnr              VARCHAR(30),    -- Stammt aus diesem Auftrag und ...  (Obsolet wegen source_dbrid?)
  beaz_agpos             SMALLINT,       -- dieser Auftragsposition            (Obsolet wegen source_dbrid?)
  beaz_belpos            INTEGER,        -- Abzuschlag bezieht sich auf diese Rechn.Pos. (muss in Listen beruecksichtigt werden damit nicht zweimal verrechnet bei Teillieferung)
  beaz_tot               NUMERIC(12,4),
  beaz_tot_basis_w       NUMERIC(12,4),
  beaz_tot_steu          NUMERIC(12,4),
  beaz_tot_steu_basis_w  NUMERIC(12,4),
   -- System (tables__generate_missing_fields)
  dbrid                  VARCHAR(32) NOT NULL DEFAULT nextval('db_id_seq'),
  CHECK (beaz_abzuproz <> 0) -- belabzu_beaz_abzuproz_check
);

-- Indizes
    CREATE INDEX belabzu__beaz_bebnr ON belabzu(beaz_bebnr);
--

-- Kontierung für AbZuschläge aus Abzu übernehmen
CREATE OR REPLACE FUNCTION belabzu__b_iu() RETURNS TRIGGER AS $$
  DECLARE Kurs NUMERIC;
          _be_rund numeric;
          bzpos  INTEGER;
  BEGIN
    IF (tg_op='UPDATE') THEN
        IF (new.beaz_abz_id <> old.beaz_abz_id) THEN
            new.beaz_konto := abz_konto FROM abzu WHERE new.beaz_abz_id = abz_id;
        END IF;
    END IF;
    IF (tg_op='INSERT') THEN
        new.beaz_konto := abz_konto FROM abzu WHERE new.beaz_abz_id = abz_id;
    END IF;
    --
    Kurs := be_umr FROM belkopf WHERE be_bnr = new.beaz_bebnr;
    new.beaz_tot := new.beaz_anz * new.beaz_abzubetrag;
    new.beaz_tot_basis_w := new.beaz_tot * Kurs;

    -- Zuschlag gleichen Auftragsbezug wie eine Belegzeile, aber Verknüpfung zur Belegzeile fehlt ...
    IF new.beaz_agnr IS NOT NULL AND new.beaz_agpos IS NOT NULL AND new.beaz_belpos IS NULL THEN
      SELECT bz_pos INTO bzpos FROM belzeil_grund WHERE bz_be_bnr = new.beaz_bebnr AND bz_auftg = new.beaz_agnr AND bz_auftgpos = new.beaz_agpos ORDER BY bz_pos LIMIT 1;
      new.beaz_belpos := bzpos;
    END IF;

    new.beaz_tot_steu := new.beaz_tot * (1 + COALESCE(new.beaz_steuproz, 0) / 100);
    new.beaz_tot_steu_basis_w := new.beaz_tot_basis_w * (1 + COALESCE(new.beaz_steuproz, 0) / 100);
    --
    IF tsystem.settings__getbool('belzeil_rund') THEN

        SELECT be_rund INTO _be_rund FROM belkopf WHERE be_bnr = new.beaz_bebnr;
        new.beaz_tot := num_round_by_precision_rule( new.beaz_tot, _be_rund );
        new.beaz_tot_basis_w := num_round_by_precision_rule( new.beaz_tot_basis_w, _be_rund );
        --
        new.beaz_tot_steu := num_round_by_precision_rule( new.beaz_tot_steu, _be_rund );
        new.beaz_tot_steu_basis_w := num_round_by_precision_rule( new.beaz_tot_steu_basis_w, _be_rund );

    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belabzu__b_iu
    BEFORE INSERT OR UPDATE
    ON belabzu
    FOR EACH ROW
    EXECUTE PROCEDURE belabzu__b_iu();
--

-- Quell-Zuschläge mit Typ 'E' werden erledigt gesetzt bei Übernahme aus Auftrag
CREATE OR REPLACE FUNCTION belabzu__a10_iud_srcabzu_done() RETURNS TRIGGER AS $$
  BEGIN
    IF (TG_OP IN ('INSERT', 'UPDATE')) THEN
        -- Anzahlungs- und Proformarechnung sowie Gutschriften (Typ P|T|G) darf der Status Verrechnet [az_done] nicht gesetzt werden (#10135)
        IF ((SELECT be_prof FROM belkopf WHERE be_bnr = new.beaz_bebnr) IN ('P', 'T', 'G')) THEN
            RETURN new;
        END IF;

        IF (TG_OP = 'UPDATE') THEN
            -- Pos. 1 war z.Bsp. Rüstkosten und wurde einfach per F2 auf Transport umgeschrieben
            IF (old.beaz_abz_id IS DISTINCT FROM new.beaz_abz_id) THEN
                -- Rg.Nr. im Auftragszuschlag zurücksetzen wenn anderer Zuschlag ausgewählt (Unabhängig vom Zuschlagstyp)
                UPDATE auftgabzu SET
                  az_bebnr = NULL
                WHERE (old.beaz_source_dbrid = auftgabzu.dbrid) -- Rg.Zuschlag wurde aus diesem auftgabzu kopiert
                  AND (old.beaz_bebnr  = az_bebnr)              -- Auftragszuschlag bisher ohne Belegnummer, da steht also immer nur die erste Belegnummer drin.
                  AND (old.beaz_abz_id = az_abz_id);            -- Es handelt sich tatsächlich um den gleichen Zuschlag (z.Bsp. Rüstkosten)

                -- Übernommen-Flag setzen mit Standardfunktion (Nur Einmalkosten)
                IF (old.beaz_source_table = 'auftgabzu') THEN
                    PERFORM TWawi.Abzu_Set_Done(old.beaz_source_table, old.beaz_source_dbrid, FALSE, TRUE);
                END IF;

                -- Zurücksetzen, der Zuschlag hat ja gewechselt und wir zeigen nicht mehr auf den bisherigen Auftragszuschlag
                new.beaz_source_table := NULL;
                new.beaz_source_dbrid := NULL;
            END IF;
        END IF;

        -- Rg.Nr. im Auftragszuschlag setzen (Unabhängig vom Zuschlagstyp)
        UPDATE auftgabzu SET
          az_bebnr = new.beaz_bebnr
        WHERE (new.beaz_source_dbrid = auftgabzu.dbrid) -- Rg.Zuschlag wurde aus diesem auftgabzu kopiert
          AND (az_bebnr IS NULL )                       -- Auftragszuschlag bisher ohne Belegnummer, da steht also immer nur die erste Belegnummer drin.
          AND (new.beaz_abz_id = az_abz_id);            -- Es handelt sich tatsächlich um den gleichen Zuschlag (z.Bsp. Rüstkosten)

        -- Übernommen-Flag setzen mit Standardfunktion (Nur Einmalkosten)
        IF (new.beaz_source_table = 'auftgabzu') THEN
            PERFORM TWawi.Abzu_Set_Done(new.beaz_source_table, new.beaz_source_dbrid, TRUE, TRUE);
        END IF;

    ELSE -- DELETE

        -- Rg.Nr. im Auftragszuschlag zurücksetzen  (Unabhängig vom Zuschlagstyp)
        UPDATE auftgabzu SET
          az_bebnr = NULL
        WHERE (old.beaz_source_dbrid = auftgabzu.dbrid) -- Rg.Zuschlag wurde aus diesem auftgabzu kopiert
          AND (old.beaz_bebnr = az_bebnr)               -- Im Auftragszuschlag stand bisher die Belegnummer
          AND (old.beaz_abz_id = az_abz_id);            -- Es handelt sich tatsächlich um den gleichen Zuschlag (z.Bsp. Rüstkosten)

        -- Übernommen-Flag zurücksetzen mit Standardfunktion (Nur Einmalkosten)
        IF (old.beaz_source_table = 'auftgabzu') THEN
            -- TODO: Raise Notice beim wieder öffnen
            PERFORM TWawi.Abzu_Set_Done(old.beaz_source_table, old.beaz_source_dbrid, FALSE, TRUE);
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belabzu__a10_iud_srcabzu_done
    AFTER INSERT OR UPDATE OR DELETE
    ON belabzu
    FOR EACH ROW
    EXECUTE PROCEDURE belabzu__a10_iud_srcabzu_done();
--

-- Kompatibilität mit P32 die Struktur der Ab-/ Zuschläge vortäuschen
CREATE OR REPLACE FUNCTION belabzu__a_iud() RETURNS TRIGGER AS $$
  DECLARE _belabzu RECORD;
          I INTEGER;
          bebnr VARCHAR;
          beposnew VARCHAR;
          beposold VARCHAR;
          --stat VARCHAR;
  BEGIN
    IF current_user = 'syncro' AND tg_op = 'DELETE' THEN
        IF tg_op = 'DELETE' THEN
            RETURN old;
        ELSE
            RETURN new;
        END IF;
    END If;
    --
    IF tg_op = 'DELETE' THEN
        bebnr := old.beaz_bebnr;
        beposold := old.beaz_belpos;
        -- Beleg-Nr. in AuftgAbzu setzen verschoben in belabzu__a10_iud_srcabzu_done (LG, 11/2018)
    ELSE
        bebnr := new.beaz_bebnr;
        beposnew := new.beaz_belpos;
        IF tg_op = 'UPDATE' THEN
            beposold := old.beaz_belpos;
        END IF;
    END IF;
    --
    UPDATE belzeil_auftg_lif SET bz_fakt = bz_fakt WHERE bz_be_bnr = bebnr; -- Preise aktualisieren bei Änderungen der Zuschläge #11419
    --
    --SELECT be_prof INTO stat FROM belkopf WHERE be_bnr = bebnr;

    -- Beleg-Nr. in AuftgAbzu setzen verschoben in belabzu__a10_iud_srcabzu_done (LG, 11/2018)
    -- Achtung: Hier wurde früher Insert/Update Fall behandelt. Scheinbar konnte man einen Rechnungszuschlag
    --   umschreiben (z.Bsp. von Rüstkosten auf Transport) und manuell mit einer anderen Auftragsposition verknüpfen woraufhin
    --   das im Auftrag rückwärts neu verknüpft wurde. Das geht nicht mehr, da die Verknüpfung heute immer über beaz_source_table
    --   und beaz_source_dbrid läuft.

    --
    PERFORM disablemodified();
    --- #19698  Löschen ab 2024   ************************* Anfang *************************
    --- -- abzuschläge p32
    --- IF tg_op <> 'DELETE' AND NOT EXISTS(SELECT true FROM belabzu_p32stru WHERE be_abzup32_be_bnr = bebnr) THEN
    ---     INSERT INTO belabzu_p32stru(be_abzup32_be_bnr) VALUES (bebnr);
    --- END IF;
    --
    --- IF current_user <> 'syncro' THEN
    ---     UPDATE belabzu_p32stru SET
    ---       be_zunr1 = NULL, be_zupr1 = NULL,be_zuanz1 = NULL, be_zunr2 = NULL, be_zupr2 = NULL, be_zuanz2 = NULL,
    ---       be_zunr3 = NULL, be_zupr3 = NULL,be_zuanz3 = NULL, be_zunr4 = NULL, be_zupr4 = NULL, be_zuanz4 = NULL,
    ---       be_zunr5 = NULL, be_zupr5 = NULL,be_zuanz5 = NULL, be_zunr6 = NULL, be_zupr6 = NULL, be_zuanz6 = NULL
    ---     WHERE be_abzup32_be_bnr = bebnr;
    --- END IF;
    --
    --- I := 1;
    --- -- 1
    --- FOR _belabzu IN SELECT * FROM belabzu WHERE beaz_bebnr = bebnr ORDER BY dbrid LIMIT 1 LOOP
    ---     UPDATE belabzu_p32stru SET be_zunr1 = _belabzu.beaz_abz_id, be_zupr1 = _belabzu.beaz_abzubetrag * _belabzu.beaz_anz, be_zuanz1 = _belabzu.beaz_anz, be_zuauftg1 = _belabzu.beaz_agnr ||_belabzu.beaz_agpos ||COALESCE(_belabzu.beaz_belpos::VARCHAR, '') WHERE be_abzup32_be_bnr = _belabzu.beaz_bebnr;
    --- END LOOP;
    --- -- 2
    --- FOR _belabzu IN SELECT * FROM belabzu WHERE beaz_bebnr = bebnr ORDER BY dbrid OFFSET 1 LIMIT 1 LOOP
    ---     UPDATE belabzu_p32stru SET be_zunr2 = _belabzu.beaz_abz_id, be_zupr2 = _belabzu.beaz_abzubetrag * _belabzu.beaz_anz, be_zuanz2 = _belabzu.beaz_anz, be_zuauftg2 = _belabzu.beaz_agnr ||_belabzu.beaz_agpos ||COALESCE(_belabzu.beaz_belpos::VARCHAR, '') WHERE be_abzup32_be_bnr = _belabzu.beaz_bebnr;
    --- END LOOP;
    --- -- 3
    --- FOR _belabzu IN SELECT * FROM belabzu WHERE beaz_bebnr = bebnr ORDER BY dbrid OFFSET 2 LIMIT 1 LOOP
    ---     UPDATE belabzu_p32stru SET be_zunr3 = _belabzu.beaz_abz_id, be_zupr3 = _belabzu.beaz_abzubetrag * _belabzu.beaz_anz, be_zuanz3 = _belabzu.beaz_anz, be_zuauftg3 = _belabzu.beaz_agnr ||_belabzu.beaz_agpos ||COALESCE(_belabzu.beaz_belpos::VARCHAR, '') WHERE be_abzup32_be_bnr = _belabzu.beaz_bebnr;
    --- END LOOP;
    --- -- 4
    --- FOR _belabzu IN SELECT * FROM belabzu WHERE beaz_bebnr = bebnr ORDER BY dbrid OFFSET 3 LIMIT 1 LOOP
    ---     UPDATE belabzu_p32stru SET be_zunr4 = _belabzu.beaz_abz_id, be_zupr4 = _belabzu.beaz_abzubetrag * _belabzu.beaz_anz, be_zuanz4 = _belabzu.beaz_anz, be_zuauftg4 = _belabzu.beaz_agnr ||_belabzu.beaz_agpos ||COALESCE(_belabzu.beaz_belpos::VARCHAR, '') WHERE be_abzup32_be_bnr = _belabzu.beaz_bebnr;
    --- END LOOP;
    --- -- 5
    --- FOR _belabzu IN SELECT * FROM belabzu WHERE beaz_bebnr = bebnr ORDER BY dbrid OFFSET 4 LIMIT 1 LOOP
    ---     UPDATE belabzu_p32stru SET be_zunr5 = _belabzu.beaz_abz_id, be_zupr5 = _belabzu.beaz_abzubetrag * _belabzu.beaz_anz, be_zuanz5 = _belabzu.beaz_anz, be_zuauftg5 = _belabzu.beaz_agnr ||_belabzu.beaz_agpos ||COALESCE(_belabzu.beaz_belpos::VARCHAR, '') WHERE be_abzup32_be_bnr = _belabzu.beaz_bebnr;
    --- END LOOP;
    --- -- 6
    --- FOR _belabzu IN SELECT * FROM belabzu WHERE beaz_bebnr = bebnr ORDER BY dbrid OFFSET 5 LIMIT 1 LOOP
    ---     UPDATE belabzu_p32stru SET be_zunr6 = _belabzu.beaz_abz_id, be_zupr6 = _belabzu.beaz_abzubetrag * _belabzu.beaz_anz, be_zuanz6 = _belabzu.beaz_anz, be_zuauftg6 = _belabzu.beaz_agnr ||_belabzu.beaz_agpos ||COALESCE(_belabzu.beaz_belpos::VARCHAR, '') WHERE be_abzup32_be_bnr = _belabzu.beaz_bebnr;
    --- END LOOP;
    --- #19698  Löschen ab 2024   ************************* Ende *************************

    IF NOT current_user = 'syncro' THEN
        -- aktualisierung von bz_tot
        IF beposold IS NOT NULL AND beposold IS DISTINCT FROM beposnew THEN
            UPDATE belzeil_grund SET bz_id = bz_id WHERE bz_be_bnr = bebnr AND bz_pos = beposold;
        END IF;
        --
        IF beposnew IS NOT NULL THEN
            IF TG_OP = 'UPDATE' THEN
                UPDATE belzeil_grund SET bz_id = bz_id WHERE bz_be_bnr = bebnr AND bz_pos = beposnew
                  AND NOT (new.beaz_abzubetrag <> old.beaz_abzubetrag AND new.beaz_abzuproz IS NOT NULL);
            ELSE
                UPDATE belzeil_grund SET bz_id = bz_id WHERE bz_be_bnr = bebnr AND bz_pos = beposnew;
            END IF;
        END IF;
        --
        UPDATE belkopf SET be_gesamt_net = COALESCE(bel_gesamt_betrag(bebnr), 0), be_gesamt_steu = COALESCE(bel_gesamt_steu(be_bnr), 0) WHERE be_bnr = bebnr;
    END IF;
    --
    PERFORM enablemodified();

    -- Meldung ausgeben,wennn Steuersatz ungültig
    IF NOT (tg_op = 'DELETE') THEN
        IF new.beaz_steu = (SELECT steu_z FROM steutxt WHERE steu_z = new.beaz_steu AND steu_valid_to < current_date) THEN
            PERFORM PRODAT_TEXT(15810);
        END IF;
    END IF;

    IF tg_op = 'DELETE' THEN
        RETURN old;
    ELSE
        RETURN new;
    END IF;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belabzu__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belabzu
    FOR EACH ROW
    EXECUTE PROCEDURE belabzu__a_iud();
--

--- #19698  Löschen ab 2024   ************************* Anfang *************************
--- CREATE TABLE belabzu_p32stru (
---   be_abzup32_be_bnr   VARCHAR(15) REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE,
---   be_zunr1            VARCHAR(40),  --Ab- und Zuschläge
---   be_zunr2            VARCHAR(40),
---   be_zunr3            VARCHAR(40),
---   be_zunr4            VARCHAR(40),
---   be_zunr5            VARCHAR(40),
---   be_zunr6            VARCHAR(40),
---   be_zupr1            NUMERIC(12,2),
---   be_zupr2            NUMERIC(12,2),
---   be_zupr3            NUMERIC(12,2),
---   be_zupr4            NUMERIC(12,2),
---   be_zupr5            NUMERIC(12,2),
---   be_zupr6            NUMERIC(12,2),
---   be_zuanz1           NUMERIC,
---   be_zuanz2           NUMERIC,
---   be_zuanz3           NUMERIC,
---   be_zuanz4           NUMERIC,
---   be_zuanz5           NUMERIC,
---   be_zuanz6           NUMERIC,
---   be_zuauftg1         VARCHAR(40),
---   be_zuauftg2         VARCHAR(40),
---   be_zuauftg3         VARCHAR(40),
---   be_zuauftg4         VARCHAR(40),
---   be_zuauftg5         VARCHAR(40),
---   be_zuauftg6         VARCHAR(40)
--- );
--- #19698  Löschen ab 2024   ************************* Ende *************************

-- Basistabelle Belegzeilen
CREATE TABLE belzeil (
  bz_id               SERIAL PRIMARY KEY,
  bz_be_bnr           VARCHAR(15) NOT NULL REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE,
  bz_pos              SMALLINT,
  bz_zubez            TEXT,
  bz_zubez_rtf        TEXT,
  -- System (tables__generate_missing_fields)
  dbrid               VARCHAR(32) NOT NULL DEFAULT nextval('db_id_seq'),
  insert_date         DATE,
  insert_by           VARCHAR(32),
  modified_by         VARCHAR(32),
  modified_date       TIMESTAMP(0)
 );

 -- Trigger
 -- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
 CREATE TRIGGER belzeil_set_modified
  BEFORE INSERT OR UPDATE
  ON belzeil
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();
 --

 --
 CREATE OR REPLACE FUNCTION belzeil__a_iud() RETURNS TRIGGER AS $$
  DECLARE bebnr VARCHAR;
  BEGIN
    IF (current_user='syncro') THEN
        RETURN new;
    END IF;

    IF TG_OP='INSERT' THEN
        bebnr:=new.bz_be_bnr;
    ELSE
        bebnr:=old.bz_be_bnr;
    END IF;

    UPDATE belkopf SET be_gesamt_net=COALESCE(bel_gesamt_betrag(be_bnr), 0), be_gesamt_steu=COALESCE(bel_gesamt_steu(be_bnr), 0) WHERE be_bnr=bebnr;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belzeil__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belzeil
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil__a_iud();
 --
--

--
CREATE OR REPLACE FUNCTION belzeil_add_auftg_pos(agnr VARCHAR, agpos INTEGER) RETURNS VARCHAR(50) AS $$
  BEGIN
    RETURN agnr||'~'||COALESCE(CAST(agpos AS VARCHAR),'1');--keine Position wird auf 1 gebucht um die zuordnung von abschlags und schlußrechnung noch zu finden
  END $$ LANGUAGE plpgsql IMMUTABLE;
--

--
CREATE OR REPLACE FUNCTION belzeil_get_auftgmainpos(bzid INTEGER, vkptotalpos BOOL) RETURNS INTEGER AS $$
  DECLARE r  RECORD;
          r1 RECORD;
          r2 RECORD;
  BEGIN
    SELECT bz_be_bnr, bz_auftg, bz_auftgpos, bz_auftghpos INTO r FROM belzeil_grund WHERE bz_id=bzid;
    FOR r1 IN SELECT bz_id, bz_auftghpos, bz_vkptotalpos FROM belzeil_grund WHERE bz_be_bnr=r.bz_be_bnr AND bz_auftg=r.bz_auftg AND bz_auftgpos=r.bz_auftghpos LOOP
        --
        IF vkptotalpos AND r1.bz_vkptotalpos THEN
                RETURN  r1.bz_id;
        ELSIF r1.bz_auftghpos IS NOT NULL THEN
                FOR r2 IN SELECT * FROM belzeil_get_auftgmainpos(r1.bz_id, vkptotalpos) LOOP
                        RETURN  r2.belzeil_get_auftgmainpos;
                END LOOP;
        ELSE --ag_hpos IS  keine weitere Position obendrüber
                IF NOT vkptotalpos THEN
                        RETURN  r1.bz_id;
                END IF;
                IF vkptotalpos AND r1.bz_vkptotalpos THEN
                        RETURN  r1.bz_id;
                END IF;
        END IF;
    END LOOP;
    RETURN null;
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION belzeil_do_auftgsubpos(bzid INTEGER, onlyvkptotalpos BOOL) RETURNS SETOF INTEGER AS $$
  DECLARE r  RECORD;
          r1 RECORD;
          r2 RECORD;
  BEGIN
    SELECT bz_be_bnr, bz_auftg, bz_auftgpos, bz_auftghpos INTO r FROM belzeil_grund WHERE bz_id=bzid;
    FOR r1 IN SELECT bz_id, bz_vkptotalpos FROM belzeil_grund WHERE bz_auftg=r.bz_auftg AND bz_auftghpos=r.bz_auftgpos AND bz_be_bnr=r.bz_be_bnr LOOP
        RETURN NEXT r1.bz_id;
        IF onlyvkptotalpos AND r1.bz_vkptotalpos THEN
        ELSE
                FOR r2 IN SELECT * FROM belzeil_do_auftgsubpos(r1.bz_id, onlyvkptotalpos) LOOP
                        RETURN NEXT r2.belzeil_do_auftgsubpos;
                END LOOP;
        END IF;
    END LOOP;
    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- virtuelle Tabelle als Grundlage für freie und Auftragsgebundene Belgzeilen, stellt u.a Menge verkauft und Steuer bereit
CREATE TABLE belzeil_grund (
  bz_zeko             VARCHAR(12), --Kontierung (Zeilenkonto)
  bz_auftg            VARCHAR(30),
  bz_auftgpos         INTEGER,
  bz_auftghpos        INTEGER,
  bz_canrabatt        BOOL NOT NULL DEFAULT TRUE,
  bz_lfs              VARCHAR(30),
  bz_bz_be_bnr        VARCHAR(15), -- Bezug zu Rechnungsposition, z.B. Originalrechung bei Gutschrift aus Rechnung
  bz_bz_pos           SMALLINT,
  -- bz_lagabgid_obsolet         INTEGER REFERENCES lifsch ON UPDATE CASCADE ON DELETE SET NULL, -- altes Feld konsistent halten, aber nicht restriktiv
  bz_belp_id          INTEGER, --constraints:  REFERENCES belegpos ON UPDATE CASCADE
  bz_aknr             VARCHAR(40),
  bz_akbz             VARCHAR(100),
  bz_ep_netto         NUMERIC(12,4),  -- Einzelpreis ohne Abzuschläge inkl. Rabatte
  bz_tot              NUMERIC(16,4),  -- bz_ep_netto inkl abzu
  bz_tot_basis_w      NUMERIC(16,4),
  bz_tot_steu         NUMERIC(16,4),  -- bz_brutto
  bz_tot_steu_basis_w NUMERIC(20,8),
  bz_fakt             NUMERIC(14,6),
  bz_fakt_uf1         NUMERIC(20,8),  --keine Längenbegrenzung wegen Rundungsfehlern!
  bz_vkp_brutto       NUMERIC(16,4),  --für Eingabe bei Kasse (Registrierkasse KTI)
  bz_vkp_uf1          NUMERIC(20,8),  --keine Längenbegrenzung wegen Rundungsfehlern!
  bz_vkp_uf1_basis_w  NUMERIC(20,8),  --keine Längenbegrenzung wegen Rundungsfehlern!
  bz_mce              INTEGER CONSTRAINT xtt4064__bz_mce REFERENCES artmgc,
  bz_uf               NUMERIC(12,4),
  bz_mcbez            VARCHAR(50),
  bz_preis            NUMERIC(12,4),                    -- Neu ab 01/2016, http://redmine.prodat-sql.de/issues/6447
  bz_preiseinheit     NUMERIC(12,4) NOT NULL DEFAULT 1 CONSTRAINT belzeil_grund__chk__preiseinheit CHECK ( bz_preiseinheit > 0), -- Bezugsgröße von bz_preis,
  bz_vkp              NUMERIC(20,8), -- Preis in Belegwährung, Beleg-ME, ohne Rabatte und dergleichen. Intern, verrechnet mit Preiseinheit
  bz_vkp_basis_w      NUMERIC(20,8),
  bz_vkpbas           NUMERIC(12,4),
  bz_vkptotalpos      BOOL NOT NULL DEFAULT FALSE,--Wenn Verkaufsposition eine TotalPos ist (also die Unterpositionen halten den Wert) dann darf die Position in der Rechnung nicht mitgezählt werden
  bz_arab             NUMERIC(5,2), --wird im trigger immer auf 0 gesetzt
  bz_rabhint          VARCHAR(75),  --währungsrabatt
  bz_gesrab           NUMERIC(5,2),
  bz_steucode         INTEGER CONSTRAINT xtt5057__bz_steucode REFERENCES steutxt ON UPDATE CASCADE,
  bz_steuproz         NUMERIC(5,2), --wird im trigger immer auf 0 gesetzt
  bz_ks               VARCHAR(9),     --Kostenstelle (war früher bz_konto)
  bz_an_nr            VARCHAR(50) REFERENCES anl ON UPDATE CASCADE ON DELETE NO ACTION,
  bz_hwpos            VARCHAR(15),
  bz_ab_ix            INTEGER, --Index zur Nachkalkulation wegen Aufteilungsbuchungen
  bz_buchdat          DATE,    --Datum Übergabe Buchhaltung
  bz_buchdone         BOOL DEFAULT FALSE,
  bz_bda              VARCHAR(40), -- ag_bda Bestellnummer Kunde
  bz_add_auftg_pos    VARCHAR(200), -- für Index für Abschlagsrechnungen
  bz_vkp_mce       INTEGER REFERENCES artmgc
) INHERITS (belzeil); -- Zitat: "WAAAAAAAAAAAAH!", (LG, 10/2013)
-- All check constraints and not-null constraints on a parent table are automatically inherited by its children.
-- Other types of constraints (unique, primary key, and foreign key constraints) are not inherited.

CREATE VIEW belzeil_txt AS SELECT * FROM ONLY belzeil_grund;

-- Indizes
    CREATE INDEX belzeil_grund_index ON belzeil_grund(bz_be_bnr);
    CREATE INDEX belzeil_grund_bz_pos ON belzeil_grund(bz_pos);
    CREATE INDEX belzeil_grund_belp_id ON belzeil_grund(bz_belp_id);
    --CREATE INDEX belzeil_bz_lagabgid ON belzeil_grund(bz_lagabgid);
    CREATE INDEX belzeil_bz_buchdone ON belzeil_grund(bz_buchdone);

    CREATE INDEX belzeil_grund_bz_aknr ON belzeil_grund(bz_aknr);

    CREATE INDEX belzeil_grund_auftg ON belzeil_grund(bz_auftg);
    CREATE INDEX belzeil_grund_auftg_pos ON belzeil_grund(bz_auftgpos);
    CREATE INDEX belzeil_add_auftg_pos ON belzeil_grund(belzeil_add_auftg_pos(bz_auftg,bz_auftgpos));
    CREATE INDEX belzeil_grund_bz_add_auftg_pos ON belzeil_grund(bz_add_auftg_pos);
    CREATE INDEX belzeil_grund_lfs ON belzeil_grund(bz_lfs);

    CREATE INDEX belzeil_grund_bz_an_nr ON belzeil_grund(bz_an_nr);
--

-- Achtung belzeil_grund Trigger nach belzeil_auftg_lif
 CREATE OR REPLACE FUNCTION belzeil_grund__b_iu() RETURNS TRIGGER AS $$
  DECLARE r             RECORD;
          Kurs          NUMERIC;
          berund        NUMERIC;
          faktor        NUMERIC;
          abzusum       NUMERIC;
          abzusumsteu   NUMERIC;
          ragnr         VARCHAR;
          ragpos        INTEGER;
  BEGIN
      SELECT ag_nr, ag_pos INTO ragnr, ragpos FROM auftg WHERE ag_id=(SELECT ag_rahmen_ag_id FROM auftg WHERE ag_nr=new.bz_auftg AND ag_pos=new.bz_auftgpos AND ag_astat='E');

      new.bz_preis        := COALESCE(new.bz_preis,0);
      new.bz_preiseinheit := Do1If0(COALESCE(new.bz_preiseinheit,1)); -- Sollte weder NULL noch 0 sein.

      IF (tg_op = 'INSERT') THEN
          -- Kein manueller Preis angegeben, bz_vkp von irgendwoher gefüllt -> bz_preis zurückrechnen.
          IF (new.bz_preis = 0) AND (new.bz_vkp <> 0) THEN
              new.bz_preis := new.bz_vkp * new.bz_preiseinheit;
          ELSE -- bz_preis angegeben oder alles ist 0
              new.bz_vkp   := new.bz_preis / new.bz_preiseinheit;
          END IF;
      END IF;

      IF (tg_op = 'UPDATE') THEN -- Preis / PE geändert -> VKP Updaten
          IF (new.bz_preis IS DISTINCT FROM old.bz_preis) OR (new.bz_preiseinheit IS DISTINCT FROM old.bz_preiseinheit) THEN
              new.bz_vkp   := new.bz_preis / new.bz_preiseinheit;
          ELSIF (new.bz_vkp <> old.bz_vkp) THEN -- VKP geändert und Preis / PE gleich -> Preis zurückrechnen.
              new.bz_preis := new.bz_vkp * new.bz_preiseinheit;
          END IF;
      END IF;

      new.bz_add_auftg_pos:=belzeil_add_auftg_pos(COALESCE(ragnr, new.bz_auftg), COALESCE(ragpos, new.bz_auftgpos));
      new.bz_arab:=COALESCE(new.bz_arab, 0);
      new.bz_gesrab:=COALESCE(new.bz_gesrab, 0);
      new.bz_steuproz:=COALESCE(new.bz_steuproz, 0);--umgerechnete Mengen und Preise

      IF new.bz_mce IS NOT NULL THEN
          new.bz_fakt_uf1 := tartikel.me__menge__in__menge_uf1(new.bz_mce, new.bz_fakt);
          new.bz_vkp_uf1 := tartikel.me__preis__in__preis_uf1(COALESCE(new.bz_vkp_mce, new.bz_mce), new.bz_vkp);
      ELSE -- freie Belegzeile ohne Real-Artikel
          new.bz_fakt_uf1 := new.bz_fakt;
          new.bz_vkp_uf1  := new.bz_vkp;

          -- Standard-ME-Bez des Artikels eintragen, Fallback auf Stück
          IF new.bz_mcbez IS NULL THEN
              new.bz_mcbez := COALESCE(prodat_languages.standard_mgc(new.bz_aknr), prodat_languages.lang_mgcode(1));
          END IF;
      END IF;

      --
      Kurs := be_umr FROM belkopf WHERE be_bnr = new.bz_be_bnr;
      new.bz_vkp_basis_w     := new.bz_vkp     * Kurs;
      new.bz_vkp_uf1_basis_w := new.bz_vkp_uf1 * Kurs;
      new.bz_vkpbas          := ak_vkpbas FROM art WHERE ak_nr=new.bz_aknr;
      --

      /*------------Positionssummen für Rechnung berechnen------------------------*/

      faktor:=IFTHEN(new.bz_canrabatt, (1-new.bz_arab/100)*COALESCE(1-new.bz_gesrab/100, 1), 1);
      new.bz_ep_netto:=new.bz_vkp*faktor;
      -- new.bz_ep_netto := tartikel.me__preis__in__preis_uf1(new.bz_vkp_mce, new.bz_vkp) * faktor;

      --prozentuale Abzu neu berechnen
      UPDATE belabzu SET
        beaz_abzubetrag = COALESCE(new.bz_ep_netto * beaz_abzuproz/100, 0) -- verwendet rabatierten Preis
      WHERE beaz_bebnr = new.bz_be_bnr AND beaz_belpos = new.bz_pos
        AND beaz_abzuproz IS NOT NULL
        AND beaz_abzubetrag <> ROUND(COALESCE(new.bz_ep_netto * beaz_abzuproz/100, 0), 4);-- Rundungsabweichungen einbeziehen, NUMERIC(12,4)

      --Abzüge/Zuschläge = (Anzahl * Betrag * [Steuerprozent])
      SELECT SUM(beaz_tot), SUM(beaz_tot_steu) INTO abzusum, abzusumsteu FROM belabzu WHERE beaz_bebnr=new.bz_be_bnr AND beaz_belpos=new.bz_pos;

      --Positionssummen
      new.bz_tot:=  (new.bz_fakt_uf1 * new.bz_vkp_uf1 * faktor) + COALESCE(abzusum, 0);     -- #7326

      new.bz_tot_steu         := round((new.bz_fakt_uf1 * new.bz_vkp_uf1 * faktor),2) /*=bz_tot. Rundung wegen Belegausgabe: "bel_gesamt_steu => siehe dortige Formel!"*/ * (1 + COALESCE(new.bz_steuproz,0)/100) + COALESCE(abzusumsteu, 0);
      new.bz_tot_basis_w      :=  new.bz_tot * Kurs;
      new.bz_tot_steu_basis_w :=  new.bz_tot_steu * Kurs;

      /*------------ Ende Positionssummen ----------------------------------------*/

      IF (tg_op='UPDATE') THEN -- Änderung Lieferschein-Vorgänger
          IF old.bz_belp_id IS DISTINCT FROM new.bz_belp_id THEN
              new.bz_lfs := beld_dokunr FROM belegdokument JOIN belegpos ON belp_dokument_id = beld_id WHERE belp_id = new.bz_belp_id LIMIT 1;
          END IF;
      ELSE
          IF new.bz_belp_id IS NOT NULL THEN
              new.bz_lfs := beld_dokunr FROM belegdokument JOIN belegpos ON belp_dokument_id = beld_id WHERE belp_id = new.bz_belp_id LIMIT 1;
          END IF;
      END IF;

      --
      IF TSystem.Settings__GetBool('belzeil_rund') THEN
          SELECT be_rund INTO berund FROM belkopf WHERE be_bnr=new.bz_be_bnr;
          IF 0.05=berund THEN
              new.bz_tot:=ROUND(ROUND(new.bz_tot*20)/20,2);
              new.bz_tot_basis_w:=ROUND(ROUND(new.bz_tot_basis_w*20)/20,2);
              --
              new.bz_tot_steu:=ROUND(ROUND(new.bz_tot_steu*20)/20,2);
              new.bz_tot_steu_basis_w:=ROUND(ROUND(new.bz_tot_steu_basis_w*20)/20,2);
          END IF;
          IF 0.01=berund THEN
              new.bz_tot:=ROUND(new.bz_tot,2);
              new.bz_tot_basis_w:=ROUND(new.bz_tot_basis_w,2);
              --
              new.bz_tot_steu:=ROUND(new.bz_tot_steu,2);
              new.bz_tot_steu_basis_w:=ROUND(new.bz_tot_steu_basis_w,2);
          END IF;
      END IF;
      --steuerfrei
      IF (COALESCE(new.bz_steucode, 0)=0) THEN
          new.bz_steuproz:=0;
          RETURN new;
      END IF;
      --
      IF current_user='syncro' THEN
          RETURN new;
      END IF;
      --steuerbehaftet
      SELECT be_steucode1, be_steucode2 INTO r FROM belkopf WHERE be_bnr=new.bz_be_bnr;
      IF (COALESCE(new.bz_steucode,0)<>COALESCE(r.be_steucode1,0))AND(COALESCE(new.bz_steucode,0)<>COALESCE(r.be_steucode2,0)) THEN
          RAISE EXCEPTION 'xtt6260 steuer_not_in_belkopf %', new.bz_be_bnr;
      END IF;

      -- Erlöskonto aus Artikelstamm
      IF TG_OP = 'INSERT' THEN
          IF new.bz_zeko IS NULL THEN
              new.bz_zeko := ak_zeko FROM art WHERE ak_nr = new.bz_aknr;
          END IF;
      ELSE -- UPDATE
          IF new.bz_zeko IS NULL AND new.bz_aknr IS DISTINCT FROM old.bz_aknr THEN
              new.bz_zeko := ak_zeko FROM art WHERE ak_nr = new.bz_aknr;
          END IF;
      END IF;

      RETURN new;
  END $$ LANGUAGE plpgsql;
 --
--ACHTUNG AUCH AN belzeil_auftg_lif
  CREATE TRIGGER belzeil_grund__b_iu
    BEFORE INSERT OR UPDATE
    ON belzeil_grund
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil_grund__b_iu();
--


-- Lieferschein-gebundene Belegzeilen
CREATE TABLE belzeil_auftg_lif (
) INHERITS (belzeil_grund);

-- Constraints
    ALTER TABLE belzeil_auftg_lif ADD CONSTRAINT bz_be_bnr_fkey FOREIGN KEY (bz_be_bnr) REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE;

    ALTER TABLE belzeil_auftg_lif ADD CONSTRAINT xtt5000__bz_aknr FOREIGN KEY (bz_aknr) REFERENCES art ON UPDATE CASCADE;
    ALTER TABLE belzeil_auftg_lif ADD CONSTRAINT xtt5057__bz_steucode FOREIGN KEY (bz_steucode) REFERENCES steutxt ON UPDATE CASCADE;
    -- ALTER TABLE belzeil_auftg_lif ADD FOREIGN KEY (bz_lagabgid_obsolet) REFERENCES lifsch ON UPDATE CASCADE ON DELETE SET NULL; -- altes Feld konsistent halten, aber nicht restriktiv
    ALTER TABLE belzeil_auftg_lif ADD CONSTRAINT xtt4064__bz_mce FOREIGN KEY (bz_mce) REFERENCES artmgc;
    ALTER TABLE belzeil_auftg_lif ADD CONSTRAINT xtt4064__bz_vkp_mce FOREIGN KEY (bz_vkp_mce) REFERENCES artmgc;
    ALTER TABLE belzeil_auftg_lif ADD CONSTRAINT bz_an_nr_fkey FOREIGN KEY (bz_an_nr) REFERENCES anl ON UPDATE CASCADE ON DELETE NO ACTION;

    ALTER TABLE belzeil_auftg_lif ALTER COLUMN bz_mce SET NOT NULL;
    ALTER TABLE belzeil_auftg_lif ALTER COLUMN bz_fakt SET NOT NULL;
    ALTER TABLE belzeil_auftg_lif ALTER COLUMN bz_aknr SET NOT NULL;
--

-- Indizes
    CREATE UNIQUE INDEX belzeil_auftg_lif_bz_id ON belzeil_auftg_lif(bz_id);
    CREATE INDEX belzeil_auftg_lif_bz_bnr ON belzeil_auftg_lif(bz_be_bnr);
    CREATE INDEX belzeil_auftg_lif_bz_pos ON belzeil_auftg_lif(bz_pos);
    CREATE INDEX belzeil_auftg_lif_belp_id ON belzeil_auftg_lif(bz_belp_id);
    --CREATE INDEX belzeil_auftg_lif_bz_lagabgid ON belzeil_auftg_lif(bz_lagabgid);

    CREATE INDEX belzeil_auftg_lif_bz_aknr ON belzeil_auftg_lif(bz_aknr);

    CREATE INDEX belzeil_auftg_lif_auftg ON belzeil_auftg_lif(bz_auftg);
    CREATE INDEX belzeil_auftg_lif_auftg_pos ON belzeil_auftg_lif(bz_auftgpos);
    CREATE INDEX belzeil_auftg_lif_bz_add_auftg_pos ON belzeil_auftg_lif(bz_add_auftg_pos);

    CREATE INDEX belzeil_auftg_lif_bz_an_nr ON belzeil_auftg_lif(bz_an_nr);
--

--

CREATE OR REPLACE FUNCTION belzeil_auftg_lif__b_01_iu__aknr() RETURNS TRIGGER AS $$
    BEGIN
        -- INSERT
        IF new.bz_mce IS NULL THEN --Standard ME für Artikel
           new.bz_mce := tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(new.bz_aknr);
        END IF;
        --
        IF tg_op = 'UPDATE' THEN
           -- die meid hat sich nicht geändert, obwohl sich der Artikel geändert hat!
           IF (new.bz_aknr IS DISTINCT FROM old.bz_aknr) AND (new.bz_mce IS NOT DISTINCT FROM old.bz_mce)
           THEN /*wir müssen natürlich auch den Mengencode mit nachziehen!*/
              new.bz_mce := tartikel.me__convertme_for_art__by__mid(new.bz_aknr, old.bz_mce, true);
           END IF;

           IF (new.bz_aknr IS DISTINCT FROM old.bz_aknr) AND (new.bz_vkp_mce IS NOT null) AND (new.bz_mce IS NOT DISTINCT FROM old.bz_vkp_mce)
           THEN /*wir müssen natürlich auch den Mengencode mit nachziehen!*/
              new.bz_vkp_mce := tartikel.me__convertme_for_art__by__mid(new.bz_aknr, old.bz_vkp_mce, false);
           END IF;
        END IF;
        --

        -- Der Datensatz wurde mit einer für den Artikel ungültigen Mengeneinheit gespeichert. Tabelle: %, ID: %, Feld: %, ART-Nr.: %, Artmgc-ID: %
        IF NOT TArtikel.me__art__artmgc__m_ids__valid(new.bz_aknr, new.bz_mce) THEN
           RAISE EXCEPTION '%', Format(lang_text(13795), 'belzeil_grund', new.bz_id::VARCHAR, 'bz_mce'    , new.bz_aknr, new.bz_mce    ::VARCHAR);
        END IF;
        IF NOT TArtikel.me__art__artmgc__m_ids__valid(new.bz_aknr, new.bz_vkp_mce) THEN
           RAISE EXCEPTION '%', Format(lang_text(13795), 'belzeil_grund', new.bz_id::VARCHAR, 'bz_vkp_mce', new.bz_aknr, new.bz_vkp_mce::VARCHAR);
        END IF;
        --
        RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belzeil_auftg_lif__b_01_iu__aknr
        BEFORE INSERT OR UPDATE
        OF bz_aknr, bz_mce, bz_vkp_mce
        ON belzeil_auftg_lif
        FOR EACH ROW
        WHEN (new.bz_aknr IS NOT null) -- wenn null, kommt NOT NULL CONSTRAINT
        EXECUTE PROCEDURE belzeil_auftg_lif__b_01_iu__aknr();

--
CREATE OR REPLACE FUNCTION belzeil_auftg_lif__b_10_iu_uf1() RETURNS TRIGGER AS $$
    DECLARE kurs NUMERIC;
    BEGIN
        IF execution_code__is_disabled( _flagname => 'belzeil' ) THEN -- FUNCTION disableauftgtrigger()
          RETURN new;
        END IF;
        --
        Kurs := be_umr FROM belkopf WHERE be_bnr = new.bz_be_bnr;
        --
        new.bz_vkp_basis_w := new.bz_vkp * Kurs;
        new.bz_vkp_uf1_basis_w := tartikel.me__preis__in__preis_uf1(new.bz_mce, new.bz_vkp_uf1 * Kurs);
        new.bz_uf := m_uf FROM artmgc WHERE m_id = new.bz_mce;
        new.bz_vkpbas := ak_vkpbas FROM art WHERE ak_nr = new.bz_aknr;
        --
        RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER belzeil_auftg_lif__b_10_iu_uf1
        BEFORE UPDATE OR INSERT
        ON belzeil_auftg_lif
        FOR EACH ROW
        EXECUTE PROCEDURE belzeil_auftg_lif__b_10_iu_uf1();
    --

    -- Trigger-Funktion belzeil_grund__b_iu an belzeil_auftg_lif
    CREATE TRIGGER belzeil_auftg_lif__b_20_iu
        BEFORE INSERT OR UPDATE
        ON belzeil_auftg_lif
        FOR EACH ROW
        EXECUTE PROCEDURE belzeil_grund__b_iu();
--

--
CREATE OR REPLACE FUNCTION belzeil_auftg_lif__a_10_iud() RETURNS TRIGGER AS $$
  DECLARE _bebnr_new      varchar;
          _bebnr_old      varchar;
          prof           varchar;
          abproz         numeric;
          auftgrec       record;
          rab            numeric;
          grab           numeric;
          wuco           integer;
          proz           numeric;
          hauptfaktm     numeric;
          abzumenge      numeric;
          vkpupos        numeric(16,4);
          anzupos        numeric(16,4);
          rundabweichung numeric(16,4);
          abzusum        numeric(12,4);
          _stkf_new       numeric;
          _stkf_old       numeric;
          _belkopf__gutschrift__lieferschein___never_opens bool = TSystem.Settings__GetBool('belkopf__gutschrift__lieferschein___never_opens', false);
  BEGIN
    IF (current_user='syncro') THEN
        RETURN new;
    END IF;
    IF execution_code__is_disabled( _flagname => 'belzeil' ) THEN -- FUNCTION disableauftgtrigger()
        RETURN new;
    END IF;
    --
    _bebnr_new := new.bz_be_bnr;
    _bebnr_old := old.bz_be_bnr;

    SELECT sum( coalesce( bz_fakt_uf1 * IFTHEN(be_prof = 'G', -1, 1) , 0 ) ) FILTER (WHERE _belkopf__gutschrift__lieferschein___never_opens OR be_txba <> 12) -- Rechnungskorrektur Lieferschein NICHT und dieser bleibt somit zu!
      INTO _stkf_new
      FROM belzeil_auftg_lif JOIN belkopf ON bz_be_bnr = be_bnr AND be_prof IN ('R', 'G')
     WHERE bz_belp_id = new.bz_belp_id;

    SELECT sum( coalesce( bz_fakt_uf1 * IFTHEN(be_prof = 'G', -1, 1) , 0 )  ) FILTER (WHERE _belkopf__gutschrift__lieferschein___never_opens OR be_txba <> 12) -- Rechnungskorrektur
      INTO _stkf_old
      FROM belzeil_auftg_lif JOIN belkopf ON bz_be_bnr = be_bnr AND be_prof IN ('R', 'G')
     WHERE bz_belp_id = old.bz_belp_id;

    --
    PERFORM disablemodified();
    --
    UPDATE belkopf
       SET be_gesamt_net  = coalesce(bel_gesamt_betrag(be_bnr), 0),
           be_gesamt_steu = coalesce(bel_gesamt_steu(be_bnr), 0)
     WHERE be_bnr IN (_bebnr_new, _bebnr_old);
    --
    IF ( TG_OP IN ('INSERT', 'UPDATE') ) THEN
        --Aktuellen Auftrag anpassen
        IF new.bz_belp_id IS NOT NULL THEN --Rechnung aus Lieferscheinpos.=> Belegpos-Trigger fuer LFS-Beleg setzt Menge Fakt. in Auftrag
            UPDATE belegpos
               SET belp_menge_done_gme = coalesce(_stkf_new, 0),
                   belp_erledigt = IfThen( ROUND( coalesce(_stkf_new, 0), 2)  >= ROUND(belp_menge_gme, 2), true, belp_erledigt)
             WHERE belp_id = new.bz_belp_id;
        ELSE
            PERFORM TBeleg.LFS_Update_Auftg(new.bz_auftg, new.bz_auftgpos);     --Fakturierte Menge im Auftrag anpassen
        END IF;
    ELSE
        --"Alten" Auftrag anpassen, falls umgeschrieben.
        IF   (TG_OP = 'DELETE')
          OR((TG_OP = 'UPDATE')
              AND (   (old.bz_auftg    IS DISTINCT FROM new.bz_auftg)
                   OR (old.bz_auftgpos IS DISTINCT FROM new.bz_auftgpos)

                   OR (old.bz_belp_id  IS DISTINCT FROM new.bz_belp_id)
                   )
             )
        THEN
            IF old.bz_belp_id IS NOT NULL THEN --Rechnung aus Lieferscheinpos.=> Belegpos-Trigger fuer LFS-Beleg setzt Menge Fakt. in Auftrag
                UPDATE belegpos
                   SET belp_menge_done_gme = coalesce(_stkf_old, 0),
                       belp_erledigt = IfThen( ROUND( coalesce(_stkf_old, 0),2)  >= ROUND(belp_menge_gme, 2), true, belp_erledigt)
                 WHERE belp_id = old.bz_belp_id;
            ELSE
                PERFORM TBeleg.LFS_Update_Auftg(old.bz_auftg, old.bz_auftgpos);     --Fakturierte Menge im Auftrag anpassen
            END IF;
        END IF;
    END IF;

    IF (prof = 'R') THEN
        PERFORM tplanterm.buchfak_auftbuch_art(new.bz_id); -- Aufteilungsbuchung erstellen
    END IF;

    PERFORM enablemodified();
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  -- Trigger-Funktion belzeil_auftg_lif__a_10_iud an belzeil_grund
  CREATE TRIGGER belzeil_grund__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belzeil_grund
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil_auftg_lif__a_10_iud();

  -- Trigger-Funktion belzeil_auftg_lif__a_10_iud an belzeil_auftg_lif
  CREATE TRIGGER belzeil_auftg_lif__a_10_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belzeil_auftg_lif
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil_auftg_lif__a_10_iud();
--

--Preis enthält Unterposition
 CREATE OR REPLACE FUNCTION belzeil_auftg_lif__a_u__totalposwert_hauptsubpos() RETURNS TRIGGER AS $$
  DECLARE vkpupos NUMERIC(16,4);
        anzupos NUMERIC(16,4);
        rundabweichung NUMERIC(16,4);
        mainbzid INTEGER;
        pwCalc NUMERIC(16,4);
        grab NUMERIC(16,4);
  BEGIN
    --
    IF new.bz_vkptotalpos AND NOT old.bz_vkptotalpos THEN--Position wird neu zu totalpos > wir geben unseren Wert nicht nach unten (0) sondern nehmen den Wert von Unten an
        PERFORM tfaktura.recalcmainposwert(new.bz_id);
        RETURN new;
    END IF;
    --
    IF execution_code__is_disabled( _flagname => 'belzeil_totalposwertsubpos' ) THEN --wir werden durch eine Unterposition aktualisiert, dann geben wir uns selbst nicht wieder nach unten
        --Rekursion nach oben
        IF new.bz_auftghpos IS NOT NULL THEN
            mainbzid:=belzeil_get_auftgmainpos(new.bz_id, true);
            --PERFORM PRODAT_MESSAGE('mainposchange:'||new.ag_pos||' mainbzid:'||mainbzid, 'Information');
            PERFORM tfaktura.recalcmainposwert(mainbzid);
        END IF;
        RETURN new;
    END IF;
    --
    PERFORM execution_code__disable( _flagname => 'belzeil' );
    -- Wert aller umsatzwirksamen Unterpositionen
    SELECT SUM(bz_tot), COUNT(1) INTO vkpupos, anzupos FROM belzeil_grund WHERE bz_id IN (SELECT * FROM belzeil_do_auftgsubpos(new.bz_id, true));

    -- Wert aktueller Position
    grab := ( 1 - COALESCE(new.bz_gesrab, 0) / 100) * ( 1 - COALESCE(new.bz_arab, 0) / 100);
    pwCalc := new.bz_tot / ( 1 - COALESCE(new.bz_gesrab, 0) / 100);  -- Rabatt muss bei Ermitllung der Unterpositionen berücksichtigt werden

    -- Wenn Unterpos. Preise haben, Hauptpositionspreis wert-anteilig auf Unterpositionen aufteilen und da reinschreiben
    IF COALESCE(vkpupos,0)>0 THEN
        UPDATE belzeil_grund SET bz_vkp = pwCalc / Do1If0(vkpupos) * bz_vkp, bz_preis = (pwCalc / Do1If0(vkpupos) * bz_vkp) * bz_preiseinheit  -- Preiseinheit mitnehmen
                       WHERE bz_id IN (SELECT * FROM belzeil_do_auftgsubpos(new.bz_id, true));
    ELSE -- Wenn Unterpos. keinen Preis haben, Hauptpositionspreis nach Anzahl der Unterpostionen aufteilen und da reinschreiben
        UPDATE belzeil_grund SET bz_vkp = pwCalc / Do1If0(anzupos),          bz_preis = (pwCalc / Do1If0(anzupos)) * bz_preiseinheit
                       WHERE bz_id IN (SELECT * FROM belzeil_do_auftgsubpos(new.bz_id, true));
    END IF;
    -- Neu durchrechen nach Aktualisierung
    SELECT SUM(bz_tot) INTO vkpupos FROM belzeil_grund WHERE bz_id IN (SELECT * FROM belzeil_do_auftgsubpos(new.bz_id, true));
    rundabweichung:=pwCalc-vkpupos; -- Wieder Hauptpos. - Summe(aller Unterpos) wegen Rundungsfehlern
    -- Fubar.
    IF rundabweichung<>0 THEN
        UPDATE belzeil_grund SET bz_vkp=bz_vkp+rundabweichung/Do1If0(bz_fakt), bz_preis = (bz_vkp+rundabweichung / Do1If0(bz_fakt)) * bz_preiseinheit WHERE bz_id=(SELECT bz_id FROM belzeil_do_auftgsubpos(new.bz_id, true) JOIN belzeil_grund ON bz_id=belzeil_do_auftgsubpos WHERE bz_fakt>0 ORDER BY bz_fakt LIMIT 1);
    END IF;
    --
    PERFORM execution_code__enable( _flagname => 'belzeil' );

    -- Gesamtsummen nachziehen
    UPDATE belkopf
       SET be_gesamt_net  = coalesce(bel_gesamt_betrag(be_bnr), 0),
           be_gesamt_steu = coalesce(bel_gesamt_steu(be_bnr), 0)
     WHERE be_bnr = new.bz_be_bnr;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belzeil_auftg_lif__a_u__totalposwert_hauptsubpos
    AFTER UPDATE
    OF bz_fakt, bz_vkp, bz_arab, bz_tot, bz_canrabatt, bz_vkptotalpos, bz_preis, bz_preiseinheit
    ON belzeil_auftg_lif
    FOR EACH ROW
    WHEN (new.bz_vkptotalpos)
    EXECUTE PROCEDURE belzeil_auftg_lif__a_u__totalposwert_hauptsubpos();
  CREATE TRIGGER belzeil__a_u__totalposwert_hauptsubpos
    AFTER UPDATE
    OF bz_fakt, bz_vkp, bz_arab, bz_tot, bz_canrabatt, bz_vkptotalpos, bz_preis, bz_preiseinheit
    ON belzeil_grund
    FOR EACH ROW
    WHEN (new.bz_vkptotalpos)
    EXECUTE PROCEDURE belzeil_auftg_lif__a_u__totalposwert_hauptsubpos();
 --

CREATE OR REPLACE FUNCTION belzeil_auftg_lif__a_id__picndoku( _bzg belzeil_grund, _isInsert boolean ) RETURNS void AS $$
BEGIN

  -- Triggerfunktion wurde ausgelagert wegen besserem Debugging und Testen.
  -- Funktion fügt Verschlagwortung der Rechnungsnummer an Auftragsdokumente der Rechnung hinzu.
  -- Beim Löschen der Rechnungsposition wird die Verschlagwortung nur gelöscht, wenn die gesamte Rechnung mit allen Positionen gelöscht wurde.

  IF _isInsert THEN

    -- mit dem Auftrag des Belegs verknüpfte Dokumente heraussuchen und diesen
    -- einen weiteren recnokeyword-Eintrag mit der Rechnungsnummer hinzufügen
    IF _bzg.bz_be_bnr IS NOT null AND _bzg.bz_auftg IS NOT null THEN
      INSERT INTO recnokeyword
        ( r_tablename, r_dbrid, r_kategorie, r_descr )
      SELECT
          'picndoku', r_dbrid, 'rechnr', _bzg.bz_be_bnr
      FROM recnokeyword
      JOIN picndoku ON picndoku.dbrid = r_dbrid AND (pd_doktype IN ( SELECT dt_id FROM dokutypes WHERE dt_abteilung = 'Verkauf' ) OR pd_doktype IN ( 'auftg_dok', 'auftg_nachtrag_dok', 'aufedok') )
      WHERE r_tablename = 'picndoku' AND r_kategorie = 'auftgtxt' AND r_descr = _bzg.bz_auftg
      ON CONFLICT DO NOTHING;
    END IF;

  ELSE

    -- Lösche Verknüpfungen mit Rechnungsnummer nur, wenn Rechnung vollständig gelöscht.
    DELETE FROM recnokeyword
    WHERE
          r_tablename = 'picndoku'
      AND r_kategorie = 'rechnr'
      AND r_descr = _bzg.bz_be_bnr
      AND NOT EXISTS ( SELECT 1 FROM belzeil_grund WHERE bz_auftg = _bzg.bz_auftg );

  END IF;

END $$ LANGUAGE plpgsql STRICT;
--

CREATE OR REPLACE FUNCTION belzeil_auftg_lif__a_id__picndoku() RETURNS TRIGGER AS $$
  BEGIN

    -- verschlagwortet Auftragsdokumente mit den zugehörigen Rechnungsnummern bzw. löscht diese

    IF tg_op = 'INSERT' THEN
      PERFORM belzeil_auftg_lif__a_id__picndoku( new, true );
    ELSE -- DELETE
      PERFORM belzeil_auftg_lif__a_id__picndoku( old, false );
    END IF;

    RETURN null;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belzeil_auftg_lif__a_id__picndoku
    AFTER INSERT OR DELETE
    ON belzeil_auftg_lif
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil_auftg_lif__a_id__picndoku();
--


 --Wert der Hauptposition aktualisieren, dazu auf die Hauptposition gehen und von dort aus den Wert aller Werthaltigen Unterpositionen kalkulieren, anschliessend diesen zurückschreiben in die Hauptposition
 CREATE OR REPLACE FUNCTION tfaktura.recalcmainposwert(mainbzid INTEGER) RETURNS NUMERIC(16,4) AS $$
  DECLARE vkpupos NUMERIC(16,4);
          anzupos NUMERIC(16,4);
          abzunetto NUMERIC(16,4);
          vkptotalpos BOOL;
          result NUMERIC(16,4);
  BEGIN
    IF COALESCE(mainbzid,0)=0 THEN
        RETURN 0;
    END IF;
    --
    PERFORM execution_code__disable( _flagname => 'belzeil' );
    PERFORM execution_code__disable( _flagname => 'belzeil_totalposwertsubpos' );
    --
    -- VKP-Summe und Anzahl der Unterpositionen
    SELECT SUM(bz_tot), COUNT(1) INTO vkpupos, anzupos FROM belzeil_grund WHERE bz_id IN (SELECT * FROM belzeil_do_auftgsubpos(mainbzid, true));

    -- Abzuschläge der hauptposition abziehen
    abzunetto:=SUM(beaz_tot) FROM belabzu JOIN belzeil_grund ON beaz_bebnr=bz_be_bnr AND beaz_belpos=bz_pos WHERE bz_id=mainbzid;
    vkpupos:=vkpupos-COALESCE(abzunetto,0);
    --
    --PERFORM PRODAT_MESSAGE('tauftg.recalcmainposwert:'||mainbzid||' Wert:'||vkpupos, 'Information');
    --
    UPDATE belzeil_grund SET
      bz_vkp   = (COALESCE(vkpupos,0)/IFTHEN(bz_canrabatt, (1-bz_arab/100), 1) / bz_fakt),
      bz_preis = (COALESCE(vkpupos,0)/IFTHEN(bz_canrabatt, (1-bz_arab/100), 1) / bz_fakt) * bz_preiseinheit
    WHERE bz_id = mainbzid AND bz_fakt>0 RETURNING bz_vkp INTO result;

    --
    PERFORM execution_code__enable( _flagname => 'belzeil_totalposwertsubpos' );
    PERFORM execution_code__enable( _flagname => 'belzeil' );
    --
    RETURN result;
  END $$ LANGUAGE plpgsql;

 --
 CREATE OR REPLACE FUNCTION belzeil__a_iu__subposwertchangemainpos() RETURNS TRIGGER AS $$
  DECLARE mainbzid INTEGER;
  BEGIN
    --Eine Unterposition wird aus einer Hauptposition herausgezogen, Hauptposition neu berechnen
    IF tg_op='UPDATE' THEN
        IF (old.bz_auftghpos IS NOT NULL) AND (new.bz_auftghpos IS DISTINCT FROM old.bz_auftghpos) THEN
            mainbzid:=belzeil_get_auftgmainpos(new.bz_id, true, old.bz_auftghpos);
            --wenn wir schieben sich aber in der Hauptpostion nichts ändert, wir ändern zB nur eine Reihenfolge
            IF mainbzid=belzeil_get_auftgmainpos(new.bz_id, true) THEN
                RETURN new;
            END IF;
            PERFORM tfaktura.recalcmainposwert(mainbzid);
        END IF;
    END IF;
    --Eine Unterposition wird in eine Hauptposition hineingezogen, oder sie ändert ihren Wert oder wird storniert
    IF new.bz_auftghpos IS NOT NULL THEN
        mainbzid:=belzeil_get_auftgmainpos(new.bz_id, true);
        --PERFORM PRODAT_MESSAGE('subposchange:'||new.ag_pos||' mainbzid:'||mainbzid, 'Information');
        PERFORM tfaktura.recalcmainposwert(mainbzid);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belzeil__a_iu__subposwertchangemainpos
    AFTER INSERT OR UPDATE
    OF bz_auftghpos, bz_fakt, bz_vkp, bz_arab, bz_tot, bz_canrabatt, bz_preis, bz_preiseinheit
    ON belzeil_grund
    FOR EACH ROW
    --WHEN (NOT new.ag_vkptotalpos) NICHT, da ich eine "Preis enthält Unterposition" - Position in eine andere schieben kann, wodurch sich diese ändern muß
    EXECUTE PROCEDURE belzeil__a_iu__subposwertchangemainpos();

  CREATE TRIGGER belzeil_auftg_lif__a_iu__subposwertchangemainpos
    AFTER INSERT OR UPDATE
    OF bz_auftghpos, bz_fakt, bz_vkp, bz_arab, bz_tot, bz_canrabatt, bz_preis, bz_preiseinheit
    ON belzeil_auftg_lif
    FOR EACH ROW
    --WHEN (NOT new.ag_vkptotalpos) NICHT, da ich eine "Preis enthält Unterposition" - Position in eine andere schieben kann, wodurch sich diese ändern muß
    EXECUTE PROCEDURE belzeil__a_iu__subposwertchangemainpos();
 --
-- Ende: Preis enthält Unterposition

-- Automatische Löschung des Lagerabgangs bei Rechnung im vereinfachten Konsignationsprozess, siehe #8093.
CREATE OR REPLACE FUNCTION belzeil_auftg_lif__b_60_d__konsignationslager() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (old.bz_belp_id IS NOT NULL) -- LFS vorhanden

    -- Funktionalität nicht aktiv, dann raus
    IF NOT TSystem.Settings__GetBool('EntnahmeVomKonsiLiefsch') THEN
            RETURN old;
    END IF;

    -- Nur mit Lieferscheinen mit Status K arbeiten.
    IF EXISTS(SELECT true FROM lieferschein_pos WHERE belp_id = old.bz_belp_id AND 'K' IN (belp_bstat, belp_bstat1, belp_bstat2)) THEN
        IF (SELECT count(*) FROM lifsch WHERE l_belp_id = old.bz_belp_id) > 1 THEN
            PERFORM PRODAT_MESSAGE(
                lang_text(16652) || E'\n\n' || lang_text(527) || ': ' ||
                    (SELECT beld_dokunr || ', ' || lang_text(164) || ': ' || belp_pos FROM lieferschein_pos JOIN lieferschein ON beld_id = belp_dokument_id WHERE belp_id = old.bz_belp_id)
                , 'Warning'); -- Warnung, dass nicht mehrere LA auf Konsi-LFS verarbeitet werden können, inkl. Ausgabe schuldiger LFS-Pos.
        ELSE
            DELETE FROM lifsch WHERE l_bz_id = old.bz_id; -- Löschen führt automatisch zu Lagerzugang
        END IF;
    END IF;

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belzeil_auftg_lif__b_60_d__konsignationslager
    BEFORE DELETE
    ON belzeil_auftg_lif
    FOR EACH ROW
    WHEN (old.bz_belp_id IS NOT NULL) -- LFS vorhanden
    EXECUTE PROCEDURE belzeil_auftg_lif__b_60_d__konsignationslager();
-- Automatische Anlage/Korrektur von Lagerabgang bei Rechnung im vereinfachten Konsignationsprozess, siehe #8093.
CREATE OR REPLACE FUNCTION belzeil_auftg_lif__a_60_iu__konsignationslager() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (new.bz_belp_id IS NOT NULL) -- LFS vorhanden

    -- Funktionalität nicht aktiv, dann raus
    IF NOT TSystem.Settings__GetBool('EntnahmeVomKonsiLiefsch') THEN
        RETURN new;
    END IF;

    -- Nur mit Lieferscheinen mit Status K arbeiten.
    IF EXISTS(SELECT true FROM lieferschein_pos WHERE belp_id = new.bz_belp_id AND 'K' IN (belp_bstat, belp_bstat1, belp_bstat2)) THEN
        -- Nicht abgebildet: ggf. mehrere LA mit untersch. Chargen auf Konsi-Auftrag zu Konsi-LFS
        -- Auftrag => m LA1 (umbuch) mit untersch. Chargen => n LFS-Pos => n=k RECH-Pos => m(n)=l LA2 (auto)
        IF (SELECT count(*) FROM lifsch WHERE l_belp_id = new.bz_belp_id) > 1 THEN
            PERFORM PRODAT_MESSAGE(
                lang_text(16652) || E'\n\n' || lang_text(527) || ': ' ||
                    (SELECT beld_dokunr || ', ' || lang_text(164) || ': ' || belp_pos FROM lieferschein_pos JOIN lieferschein ON beld_id = belp_dokument_id WHERE belp_id = new.bz_belp_id)
                , 'Warning'); -- Warnung, dass nicht mehrere LA auf Konsi-LFS verarbeitet werden können, inkl. Ausgabe schuldiger LFS-Pos.
        ELSE -- Automatischer LA
            IF TG_OP = 'INSERT' THEN
                INSERT INTO lifsch (l_krz, l_krzl, l_krzf, l_ag_id, l_lgort, l_lgchnr, l_aknr, l_abg_mec, l_abgg, l_inliefdok, l_bz_id)
                SELECT beld_krzbesteller, beld_krzlieferung, beld_krzrechnung, belp_ag_id, konsi_lifsch.l_lgort_ue, konsi_lifsch.l_lgchnr, new.bz_aknr, new.bz_mce, new.bz_fakt, true, new.bz_id
                FROM lieferschein_pos
                  JOIN lieferschein ON beld_id = belp_dokument_id
                  JOIN lifsch AS konsi_lifsch ON konsi_lifsch.l_belp_id = belp_id -- mehrere LA auf LFS möglich
                WHERE belp_id = new.bz_belp_id;

            ELSE -- UPDATE
                IF new.bz_fakt_uf1 IS DISTINCT FROM old.bz_fakt_uf1 THEN -- Menge hat sich geändert
                    UPDATE lifsch SET
                      l_abg_mec= new.bz_mce,
                      l_abgg= new.bz_fakt
                    WHERE l_bz_id = new.bz_id;
                END IF;
            END IF;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER belzeil_auftg_lif__a_60_iu__konsignationslager
    AFTER INSERT OR UPDATE
    OF bz_fakt, bz_mce, bz_fakt_uf1
    ON belzeil_auftg_lif
    FOR EACH ROW
    WHEN (new.bz_belp_id IS NOT NULL) -- LFS vorhanden
    EXECUTE PROCEDURE belzeil_auftg_lif__a_60_iu__konsignationslager();

--
CREATE TABLE belzeil_frei (
)INHERITS (belzeil_grund);

-- Constraints
    ALTER TABLE belzeil_frei ADD CONSTRAINT bz_be_bnr_fkey FOREIGN KEY (bz_be_bnr) REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE;
    ALTER TABLE belzeil_frei ADD CONSTRAINT xtt5057__bz_steucode FOREIGN KEY (bz_steucode) REFERENCES steutxt ON UPDATE CASCADE;
    ALTER TABLE belzeil_frei ADD CONSTRAINT bz_an_nr_fkey FOREIGN KEY (bz_an_nr) REFERENCES anl ON UPDATE CASCADE ON DELETE NO ACTION;

    ALTER TABLE belzeil_frei ALTER COLUMN bz_fakt SET NOT NULL;
    ALTER TABLE belzeil_frei ALTER COLUMN bz_aknr SET NOT NULL;
--

-- Indizes
    CREATE INDEX belzeil_frei_index ON belzeil_frei(bz_be_bnr);
    CREATE INDEX belzeil_frei_bz_pos ON belzeil_frei(bz_pos);
    CREATE UNIQUE INDEX xtt5051 ON belzeil (bz_be_bnr, bz_pos);

    CREATE INDEX belzeil_frei_auftg ON belzeil_frei(bz_auftg);
    CREATE INDEX belzeil_frei_auftg_pos ON belzeil_frei(bz_auftgpos);
    CREATE INDEX belzeil_frei_bz_add_auftg_pos ON belzeil_frei(bz_add_auftg_pos);

    CREATE INDEX belzeil_frei_bz_an_nr ON belzeil_frei(bz_an_nr);
--

-- Trigger-Funktion belzeil_auftg_lif__a_10_iud an belzeil_frei
  CREATE TRIGGER belzeil_frei__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON belzeil_frei
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil_auftg_lif__a_10_iud();
--

-- Trigger-Funktion belzeil_grund__b_iu an belzeil_frei
  CREATE TRIGGER belzeil_frei__b_iu
    BEFORE INSERT OR UPDATE
    ON belzeil_frei
    FOR EACH ROW
    EXECUTE PROCEDURE belzeil_grund__b_iu();
--

--Aufteilungsbuchung für Rechnung
 CREATE TABLE belauftbuch(
   bab_id               SERIAL PRIMARY KEY,
   bab_bz_bnr           VARCHAR(15) NOT NULL REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE,
   bab_bz_pos           INTEGER NOT NULL,
   bab_ks               VARCHAR(10) NOT NULL,
   bab_netto            NUMERIC(12,2) NOT NULL,
   bab_done             BOOLEAN, -- False => Noch nicht exportiert, NULL => Noch nicht exportiert, in Oberfläche angezeigt, TRUE exportiert und abgeschlossen
   bab_rueckbuch        BOOLEAN DEFAULT FALSE --benötigt zur Unterscheidung ob Belegzeile nach Verbuchung als verbucht markiert werden soll
 );

 CREATE INDEX belauftbuch_bab_bz_bnr ON belauftbuch(bab_bz_bnr);

-- Belege - Funktionen für Gesamtbeträge -----------------------

--
CREATE OR REPLACE FUNCTION bel_gesamt_betrag(bebnr VARCHAR, noAbzu BOOLEAN DEFAULT FALSE) RETURNS NUMERIC AS $$
  DECLARE bsum        NUMERIC;
          abzusum     NUMERIC;
          f           NUMERIC;
          fAbzu       NUMERIC;
          fnoAbzu     NUMERIC;
          rec         RECORD;
          berund      NUMERIC;
          beabprozent NUMERIC;
 BEGIN
    f := 0;

    SELECT be_rund, be_abprozent
    INTO berund, beabprozent
    FROM belkopf WHERE be_bnr = bebnr;

    SELECT
      sum(round(bz_tot, 2) ),
      sum(round(bz_ep_netto * bz_fakt, 2) )
    INTO fAbzu, fnoAbzu
    FROM belzeil_grund
    WHERE bz_be_bnr = bebnr
      AND (
        NOT bz_vkptotalpos
        OR NOT EXISTS(
          SELECT true FROM belzeil_grund b1 -- Preis enthält Unterposition nur dann = 0, wenn auch Unterposition, sonst einbeziehen
          WHERE b1.bz_be_bnr = belzeil_grund.bz_be_bnr
            AND b1.bz_auftg = belzeil_grund.bz_auftg AND b1.bz_auftghpos = belzeil_grund.bz_auftgpos
        )
      )
    ;

    --ab/zuschläge noch dazu
    IF noAbzu THEN -- Summe ohne Zuschläge ausgeben
      abzusum := 0;
      f := fnoAbzu;
    ELSE
      SELECT sum(beaz_tot)
        INTO abzusum
        FROM belabzu WHERE beaz_bebnr = bebnr AND beaz_belpos IS NULL; -- die positionsbezogenen sind in bz_tot bereits eingerechnet

      f := fAbzu;
    END IF;

    bsum := ( COALESCE(f, 0) + COALESCE(abzusum, 0) ) * COALESCE(beabprozent / 100, 1); -- abschlagszahlung prozent

    IF 0.05 = berund THEN
          RETURN ROUND(ROUND(bsum * 20) / 20, 2);
    ELSE
          RETURN ROUND(bsum, 2);
    END IF;
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION belzeil_pos_wert_calc(_bzid integer, _bzstk numeric, _basis_w boolean, _brutto boolean) RETURNS numeric(16,4) AS $$
  DECLARE pospreis  numeric;
          abprozent numeric(12,4);
  BEGIN
    SELECT CASE WHEN bz_fakt * least(_bzstk, bz_fakt) <> 0 THEN
                CASE WHEN _basis_w     AND _brutto     THEN bz_tot_steu_basis_w
                     WHEN _basis_w     AND NOT _brutto THEN bz_tot_basis_w
                     WHEN NOT _basis_w AND _brutto     THEN bz_tot_steu
                     WHEN NOT _basis_w AND NOT _brutto THEN bz_tot
                END
                / bz_fakt -- Wert pro fakturierter Menge
                * least(_bzstk, bz_fakt) -- anteilige der zu berechnenden Menge
            ELSE 0
            END AS preis,
            coalesce(be_abprozent / 100, 1)
      INTO pospreis, abprozent
      FROM belzeil_grund
      JOIN belkopf ON be_bnr = bz_be_bnr
     WHERE bz_id = _bzid;
    --
      -- ACHTUNG LEAST ist für Abschlagsrechnung hier drin. Damit soll erreicht werden, das maximal der Wert der Abschlagsrechnung gutgeschrieben wird. (Rückrechnung bei Schlußrechnung)
      -- Ist die Schlußrechnung in der Menge höher als die Abschlagsrechnung, darf nicht automatisch mehr gerechnet werden!. zB es wurden 500 Stck angezahlt, aber 550 ausgeliefert.(?)
      --
      -- Weiteres bsp: es wurden 5000 Stk angezahlt und 3000 richtig ausgeliefert und schlußverrechnet, damit bleiben 2000 stk Anzahlung offen obwohl Schlußrechnung.
      -- Wir nehmen also von der Abschlagsrechnung, welche über 5000 Stk war nur den Schlußrechnugnswert von 3000 Stk als Gegenrechnung für die Schlußrechnung
    --

    RETURN round((pospreis * abprozent), 2);
  END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION belzeil_pos_wert(bzid integer) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, NULL, false, false);
  END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION belzeil_pos_wert_basis_w(bzid integer) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, NULL, true, false);
  END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION belzeil_pos_wert_steu(bzid integer) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, NULL, false, true);
  END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION belzeil_pos_wert_basis_w_steu(bzid integer) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, NULL, true, true);
  END $$ LANGUAGE plpgsql STABLE;

  --Für Abschlagsrechnung anteilig die Positionswerte bei unterschiedlichen Mengen
  CREATE OR REPLACE FUNCTION belzeil_pos_wert(bzid integer, bzstk numeric(16,2)) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, bzstk, false, false);
  END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION belzeil_pos_wert_steu(bzid integer, bzstk numeric(16,2)) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, bzstk, false, true);
  END $$ LANGUAGE plpgsql STABLE;

  CREATE OR REPLACE FUNCTION belzeil_pos_wert_basis_w_steu(bzid integer, bzstk numeric(16,2)) RETURNS numeric(16,4) AS $$
    BEGIN
      RETURN belzeil_pos_wert_calc(bzid, bzstk, true, true);
  END $$ LANGUAGE plpgsql STABLE;
--

CREATE OR REPLACE FUNCTION bel_gesamt_steu(bebnr VARCHAR) RETURNS NUMERIC AS $$
  DECLARE bsum NUMERIC;
          abzusum NUMERIC;
          f NUMERIC;
          rec RECORD;
          berund NUMERIC;
          beabprozent NUMERIC;
  BEGIN
    f := 0;

    SELECT be_rund, be_abprozent
    INTO berund, beabprozent
    FROM belkopf WHERE be_bnr = bebnr;

    SELECT
      SUM(steu_subsum.sum_per_steu)
    INTO f
    FROM (
        SELECT sum ( round( round( bz_tot, 2 ) * (100 + bz_steuproz) / 100, 2) ) AS sum_per_steu
        FROM belzeil_grund
        WHERE bz_be_bnr = bebnr
          AND (
            NOT bz_vkptotalpos
            OR NOT EXISTS(
              SELECT true FROM belzeil_grund b1 --Preis enthält Unterposition nur dann = 0, wenn auch Unterposition, sonst einbeziehen
              WHERE b1.bz_be_bnr = belzeil_grund.bz_be_bnr
                AND b1.bz_auftg = belzeil_grund.bz_auftg AND b1.bz_auftghpos = belzeil_grund.bz_auftgpos
            )
          )
        GROUP BY bz_steuproz
    ) AS steu_subsum;

    -- alt: SELECT SUM(bz_tot_steu) INTO f FROM belzeil_grund WHERE bz_be_bnr=bebnr AND NOT bz_vkptotalpos;
    --
    --Ab/Zuschläge noch dazu (auch versteuert)
    SELECT sum( beaz_tot_steu )
    INTO abzusum
    FROM belabzu
    WHERE beaz_bebnr = bebnr AND beaz_belpos IS NULL; -- die positionsbezogenen sind in bz_tot bereits eingerechnet.

    bsum := ( COALESCE(f, 0) + COALESCE(abzusum, 0) ) * COALESCE(beabprozent / 100, 1);

    IF 0.05 = berund THEN
        bsum := round(bsum * 20) / 20;
    END IF;

    RETURN round(bsum, 2);

  END $$ LANGUAGE plpgsql;
--

 /*Mahnungen
 TODO: Mahnung gedruckt - Status Feld //damit bei Auswahl eines Datumbereichs bereits gedruckte nicht auf dem Dokument erscheinen  */

CREATE TABLE belmahn
  (bm_id                SERIAL PRIMARY KEY,
   bm_be_bnr            VARCHAR(15) NOT NULL REFERENCES belkopf ON UPDATE CASCADE ON DELETE CASCADE,
   bm_date              DATE DEFAULT current_date,
   bm_ms                INTEGER NOT NULL,                       -- Mahnstufe
   bm_betrag            NUMERIC(12,4),                          -- angemahnte Summe
   bm_mahngeb           NUMERIC(12,4),                          -- Mahngebühr
   bm_verzugzins        NUMERIC(12,4),                          -- Verzugszinsen
   bm_definitiv         BOOLEAN DEFAULT FALSE,                  -- Mahnung gedruckt
   bm_apint             VARCHAR(10) DEFAULT tsystem.current_user_ll_db_usename(),       -- CONSTRAINT Ansprechpartner Intern...Standard angemeldeter Nutzer
   bm_bem               TEXT,                                   -- Bemerkung zur Mahnung
   bm_mtxt              TEXT                                    -- Mahntext
  );

 CREATE INDEX belmahn_dat ON belmahn (bm_be_bnr, bm_date);

 /*Mahnstufe und datum in belkopf eintragen*/
 CREATE OR REPLACE FUNCTION belmahn__a_iud() RETURNS TRIGGER AS $$
 DECLARE bebnr VARCHAR;
 BEGIN
  IF tg_op='DELETE' THEN
        bebnr:=old.bm_be_bnr;
  ELSE
        bebnr:=new.bm_be_bnr;
  END IF;
  --
  UPDATE belkopf
  SET be_ms=(SELECT bm_ms FROM belmahn WHERE bm_be_bnr=bebnr ORDER BY bm_ms DESC, bm_date DESC LIMIT 1),
        be_mahnd=(SELECT bm_date FROM belmahn WHERE bm_be_bnr=bebnr ORDER BY bm_ms DESC, bm_date DESC LIMIT 1)
  WHERE be_bnr=bebnr;
  --
  IF tg_op='DELETE' THEN
        RETURN old;
  ELSE
        RETURN new;
  END IF;
  --
  RETURN new;
 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER belmahn__a_iud
 AFTER UPDATE OR INSERT OR DELETE
 ON belmahn
 FOR EACH ROW
 EXECUTE PROCEDURE belmahn__a_iud();


/*Zahlung / Teilzahlung*/

CREATE TABLE teilzahl
  (tz_id                SERIAL PRIMARY KEY,
   tz_be_bnr            VARCHAR(15) NOT NULL REFERENCES belkopf ON UPDATE CASCADE,
   tz_date              DATE DEFAULT current_date,
   tz_betrag            NUMERIC(16,4) NOT NULL,
   tz_betrag_basis_w    NUMERIC(16,4),
   tz_skonto            BOOL NOT NULL DEFAULT FALSE,
   tz_zahlart           VARCHAR(50), --überweisung, scheck, ec
   tz_bem               TEXT, -- Bemerkung zur Teilzahlung
   tz_ktv_name          VARCHAR(100) REFERENCES ktovz ON UPDATE CASCADE
  );


 CREATE OR REPLACE FUNCTION teilzahl__b_iu() RETURNS TRIGGER AS $$
 BEGIN
  new.tz_betrag_basis_w:=new.tz_betrag*be_umr FROM belkopf WHERE be_bnr=new.tz_be_bnr;
  RETURN new;
 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER teilzahl__b_iu
 BEFORE INSERT OR UPDATE
 ON teilzahl
 FOR EACH ROW
 EXECUTE PROCEDURE teilzahl__b_iu();


 CREATE OR REPLACE FUNCTION bel_bezahlt(bebnr VARCHAR) RETURNS NUMERIC(16,4) AS $$
 DECLARE s NUMERIC(16,4);
 BEGIN
  SELECT SUM(tz_betrag) INTO s FROM teilzahl WHERE tz_be_bnr=bebnr;
  RETURN COALESCE(s,0);
 END $$ LANGUAGE plpgsql STABLE;


 CREATE OR REPLACE FUNCTION bel_bezahlt(bebnr VARCHAR, datb DATE) RETURNS NUMERIC(16,4) AS $$
 DECLARE s NUMERIC(16,4);
 BEGIN
  SELECT SUM(tz_betrag) INTO s FROM teilzahl WHERE tz_be_bnr=bebnr AND tz_date<=COALESCE(datb, current_date);
  RETURN COALESCE(s,0);
 END $$ LANGUAGE plpgsql STABLE;
--

-- Betrag aller vorherigen Anzahlungen bzw. Abschlägen (Teilrechnungen inkl. diesbzgl. Gutschriften) einer Schlussrechnung bzw. Rechnung.
  -- Fall 1: Bei Schlussrechnung: Kompletten Wert aller auftragsbezogenen Teilrechnungen (inkl. diesbzgl. Gutschriften) berücksichtigen.
  -- Fall 2: Bei anderen Rechnungsarten: Wert anhand der hier verrechneten Mengen von nur auftragspositionsbezogene Teilrechnung (inkl. diesbzgl. Gutschriften) berücksichtigen.
-- Verwendung: Anzeige wieviel von der Rechnung tatsächlich Zahlbetrag ist.
-- Achtung: Rechnung - Zahlbetrag <> Rechnungswert (Anzahlungs-, Abschlags-, Schlussrechnung)
-- DELETE FROM tcache.function_cache WHERE c_funcname = 'bel_abschlagrechbetrag';
CREATE OR REPLACE FUNCTION bel_abschlagrechbetrag(bebnr VARCHAR, brutto BOOL, basis_w BOOL DEFAULT FALSE) RETURNS NUMERIC(16,4) AS $$
  DECLARE current_invoice RECORD;
          result          NUMERIC(16,4);
  BEGIN
    IF bebnr IS NULL THEN RETURN 0; END IF;
    brutto := COALESCE(brutto, false); basis_w := COALESCE(basis_w, false); -- Kompatibilität zum alten ifThen(brutto, ifThen(basis_w, bz_tot_steu_basis_w, bz_tot_steu), ifThen(basis_w, bz_tot_basis_w, bz_tot))

    -- Aufbau identisch zum Zentraldokument (subDataSet 8 anzrech) und der Faktura (Tab 7 Anzahlungs-Abschlagsrechnung)

    -- Daten der zu berechnenden Rechnung
    SELECT
      be_bnr,
      be_prof,
      be_bdat,
      be_txba,
      be_txba = 10                AS is_closing_invoice,  -- aktuelle Rechnung ist Schlussrechnung: entscheidet über Fall 1 oder Fall 2
      array_agg(bz_auftg)         AS bz_auftg,            -- alle Aufträge (für Berechnung an Schlussrechnung)
      array_agg(bz_add_auftg_pos) AS bz_add_auftg_pos     -- alle Auftragspositionen (für Berechnung an anderen Rechnungsarten)
    INTO current_invoice
    FROM belkopf --kein public, da auf deprecated...
      JOIN belzeil_grund ON bz_be_bnr = be_bnr
    WHERE be_bnr = bebnr
      AND (be_prof = 'R' OR be_txba = 10) -- aktuelle Rechnung ist Rechnung oder Schlussrechnung
    GROUP BY be_bnr, be_bdat, be_txba;

    IF current_invoice.be_prof IN ('T', 'G') OR current_invoice.be_prof IS NULL THEN -- Unklar, ob kumulierte Abschlagsrechnung hier mit hinein muss. -- alt: AND NOT (txba=7)
        RETURN 0;
    END IF;

    -- caching
    SELECT c_resultn INTO result FROM tcache.function_cache WHERE c_funcname = 'bel_abschlagrechbetrag' AND c_param0 = current_invoice.be_bnr AND c_param1 = current_invoice.be_txba AND c_param2 = brutto::VARCHAR AND c_param3 = basis_w::VARCHAR AND NOT c_dirty;
    IF result IS NOT NULL THEN RETURN result; END IF;
    -- end caching

    -- Eigentliche Berechnung unterschieden anhand Schlussrechnung oder nicht.
    IF current_invoice.is_closing_invoice THEN  -- Fall 1 - Schlussrechnungen:
        -- Alle Abschlagsrechnungen an Aufträge, die in der Schlussrechnung aufgenommen sind, werden gegengerechnet (Verbindung über Auftrag).
        -- Keine anteilige Berücksichtigung von fakturierten Mengen. Der gesamte Rechnungswert wird berücksichtigt (auch Zusatzpositionen und Belegzuschläge).
        SELECT
          SUM(
            CASE WHEN brutto      AND basis_w     THEN be_gesamt_steu_basis_w
                 WHEN brutto      AND NOT basis_w THEN be_gesamt_steu
                 WHEN NOT brutto  AND basis_w     THEN be_gesamt_net_basis_w
                 WHEN NOT brutto  AND NOT basis_w THEN be_gesamt_net
                 ELSE 0
            END
            * ifthen(be_txba IN (2, 11, 12), -1, 1) -- Gutschriften und Rechnungskorrekturen ggf. abziehen
          )
        INTO result
        FROM belkopf
          LEFT JOIN LATERAL (
              -- Rechnungen in Bezug zu Gutschriften und Rechnungskorrekturen
              SELECT array_agg(bz_bz_be_bnr) AS invoice_references FROM belzeil_grund WHERE bz_be_bnr = be_bnr AND bz_bz_be_bnr IS NOT NULL
          ) AS current_references ON (be_prof = 'G' OR be_txba IN (2, 11, 12))
        WHERE
          ( -- Rechnungstypen
              (be_prof = 'T' OR be_txba IN (4, 6, 7)) -- Anzahlungs- und Abschlagsrechnungen (kumuliert)
              OR (
                  (be_prof = 'G' OR be_txba IN (2, 11, 12)) -- Gutschriften und Rechnungskorrekturen, siehe #11546
                  AND invoice_references IS NOT NULL        -- mit Bezug zu Anzahlungs- und Abschlagsrechnungen (kumuliert)
                  AND EXISTS(SELECT true FROM belkopf AS refd_inv WHERE refd_inv.be_bnr = ANY(invoice_references) AND (refd_inv.be_prof = 'T' OR refd_inv.be_txba IN (4, 6, 7)))
              )
          )
          AND EXISTS(SELECT true FROM belzeil_grund WHERE bz_be_bnr = be_bnr AND bz_auftg = ANY(current_invoice.bz_auftg)) -- an entspr. Aufträgen
          AND be_bdat <= current_invoice.be_bdat
          AND be_bnr <> current_invoice.be_bnr
        ;
    ELSE  -- Fall 2 - normale Rechnungsarten:
        -- Es werden nur die Positionen aus Teilrechnungen gegengerechnet, welche auch in der Rechnung als Position der zug. Auftragsposition aufgenommen sind (Verbindung über Auftragspositionen).
        -- Normale Rechnungsarten werden anteilig der fakturierten Menge gegengerechnet (Stichwort: Menge aus Abschlagsrechnungen)
        SELECT
          SUM(belzeil_pos_wert_calc(bz_id, sum_fakt, basis_w, brutto)
          * ifthen(be_txba IN (2, 11, 12), -1, 1)) -- Gutschriften und Rechnungskorrekturen ggf. abziehen
        INTO result
        FROM belkopf
          JOIN belzeil_grund ON bz_be_bnr = be_bnr
          LEFT JOIN LATERAL (
              -- an aktueller Rechnung fakturierte Menge der zug. Auftragsposition
              SELECT SUM(bz_fakt) AS sum_fakt FROM belzeil_grund AS bz1 WHERE bz1.bz_be_bnr = current_invoice.be_bnr AND bz1.bz_add_auftg_pos = belzeil_grund.bz_add_auftg_pos
          ) AS current_invoice_pos ON true
        WHERE
          ( -- Rechnungstypen
              (be_prof = 'T' OR be_txba IN (4, 6, 7)) -- Anzahlungs- und Abschlagsrechnungen (kumuliert)
              OR (
                  (be_prof = 'G' OR be_txba IN (2, 11, 12)) -- Gutschriften und Rechnungskorrekturen, siehe #11546
                  AND bz_bz_be_bnr IS NOT NULL              -- mit Bezug zu Anzahlungs- und Abschlagsrechnungen (kumuliert)
                  AND EXISTS(SELECT true FROM belkopf AS refd_inv WHERE refd_inv.be_bnr = bz_bz_be_bnr AND (refd_inv.be_prof = 'T' OR refd_inv.be_txba IN (4, 6, 7)))
              )
          )
          AND bz_add_auftg_pos = ANY(current_invoice.bz_add_auftg_pos) -- an zug. Auftragspositionen
          AND be_bdat <= current_invoice.be_bdat
          AND be_bnr <> current_invoice.be_bnr
        ;
    END IF;

    result := COALESCE(result, 0) * (ifThen(current_invoice.be_txba = 11, -1, 1)); -- Stornorechnung, Abschläge rückwärts rechnen

    -- caching schreiben
    PERFORM tcache.function_cache_setcache('bel_abschlagrechbetrag', current_invoice.be_bnr::VARCHAR, current_invoice.be_txba::VARCHAR, brutto::VARCHAR, basis_w::VARCHAR, NULL::VARCHAR, NULL::VARCHAR, NULL::VARCHAR, result);
    -- end caching schreiben

    RETURN result;
  END $$ LANGUAGE plpgsql STABLE;
--

/* -- veraltet?
-- VARIANTE 2: Kompletter Betrag der Anzahlung unabhängig des Auftragspositions-Bezuges mit Folgerechnung verrechnen

--holt zu einer Rechnung ihr zugeörigen Anzahlungen/Abschläge.
--Verwendung: Anzeige wieviel von der Rechnung tatsächlich Zahlbetrag ist.
--Achtung: Rechnung - Zahlbetrag <> Rechnungswert (Anzahlungsrechnung, Abschlagsrechnung, Schlußrechnung)
--DELETE FROM tcache.function_cache WHERE c_funcname='bel_abschlagrechbetrag';


CREATE OR REPLACE FUNCTION bel_abschlagrechbetrag(bebnr VARCHAR, brutto BOOL, basis_w BOOL DEFAULT FALSE) RETURNS NUMERIC(16,4) AS $$
  DECLARE r RECORD;
          n NUMERIC(16,4);
          s NUMERIC(16,4);
          prof VARCHAR(5);
          txba INTEGER;
          bdat DATE;
          schlussr BOOLEAN;
          beforerech_is_anzahl BOOLEAN;
          result NUMERIC;
  BEGIN
    SELECT be_prof, be_txba, be_bdat INTO prof, txba, bdat FROM belkopf WHERE be_bnr=bebnr;

    IF prof IN ('T', 'G') THEN --kumulierte Rechnungen rechnen sich nicht gegeneinander auf, außer sie sind -- alt: AND NOT (txba=6)
        beforerech_is_anzahl:=False;
        IF (txba=6) THEN     --kumulierte Rechnung: die before Rechnung wird dann abgezogen, wenn diese eine Anzahlungsrechnung mit Prozent ist. Ansonsten ist die Vorrechnung eine normale kum. rechnung und wird nicht abgezogen!
            SELECT COALESCE(bk2.be_abprozent,100)<>100 INTO beforerech_is_anzahl
                FROM belzeil_grund bzg1 JOIN belkopf bk2 ON bk2.be_bnr=bzg1.bz_be_bnr JOIN belkopf ON belkopf.be_bnr=bebnr
                WHERE
                    belzeil_add_auftg_pos(bzg1.bz_auftg,bzg1.bz_auftgpos) IN (SELECT belzeil_add_auftg_pos(bz_auftg,bz_auftgpos) FROM belzeil_grund WHERE bz_be_bnr=bebnr)
                    AND bk2.be_prof='T'
                    AND bk2.be_bdat<=belkopf.be_bdat
                    AND bk2.be_bnr<belkopf.be_bnr
                ORDER BY bk2.be_bnr DESC LIMIT 1;
        END IF;
        IF NOT beforerech_is_anzahl THEN
            RETURN 0;
        END IF;
    END IF;
    --
    schlussr:=txba=10;--Schlussrechnung
    --caching
    SELECT c_resultn INTO result FROM tcache.function_cache WHERE c_funcname='bel_abschlagrechbetrag' AND c_param0=bebnr AND c_param1=txba AND c_param2=brutto::VARCHAR AND c_param3=basis_w::VARCHAR AND NOT c_dirty;
    IF result IS NOT NULL THEN
        RETURN result;
    END IF;
    --end caching
    s:=0;
    --
    IF NOT schlussr THEN--normale Rechnung: es werden nur die Positionen aus Abschlagsrechnungen gegengerechnet, welche auch in der Rechnung als Position aufgenommen sind
        FOR r IN SELECT DISTINCT be_bnr FROM belkopf JOIN belzeil_grund ON bz_be_bnr=be_bnr
            WHERE
                be_prof IN ('T')
                AND (txba<>6 OR COALESCE(be_abprozent,100)<>100)  --Anzahlung Prozentual und anschliessend Kumuliert weiter. Wenn die Folgerechnung Kumuliert ist, darf die Vorrechnung (hat auch status "T") nur dann abgezogen werden, wenn diese wirklich eine Teilrechnung ist. Dies ist nur an dem Prozentwert zu erkennen!
                                                                  --Bsp: R2014-0740, 721 bei MR-SUN
                                                                  --beim Einstieg in die Funktion wird bereits geprüft, ob die Vorrechnung auch wirklich anzahlung ist, oder ob der Vorgang bereits durch eine erste kumulierte aufgehoben wurde. in dem fall wird dann oben gleich ein exit gemacht und hier nicht weitergerechnet.
                AND be_bdat<=bdat
                AND be_bnr<>bebnr
                AND bz_add_auftg_pos IN (SELECT bz_add_auftg_pos FROM belzeil_grund WHERE bz_be_bnr=bebnr)
        LOOP
            --Ausrechnen wieviel des Rechnungswerts hier abgezogen werden, daher die Positionssummen der in beiden Rechnungen enthaltenen Aufträge nehmen
            --dazu anteilig die Abschlagsrechnung nehmen (Preis/Stückzahl) und mit Stückzahl der aktuellen Rechnung multiplizieren, um die Abschlagswerte auszurechnen, wenn keine Prozentzahl
            n:=SUM(IFTHEN(brutto, IFTHEN(basis_w, bz_tot_steu_basis_w, bz_tot_steu),
                       IFTHEN(basis_w, bz_tot_basis_w, bz_tot)
                       )*COALESCE(be_abprozent/100, 1)
                       /DO1IF0(bz_fakt) --Menge der Anzahlungsrechnung
                       *NUMERIC_SMALLER(DO1IF0((SELECT SUM(bz_fakt) FROM belzeil_grund bzgr WHERE bzgr.bz_be_bnr=r.be_bnr AND bzgr.bz_add_auftg_pos=belzeil_grund.bz_add_auftg_pos)), bz_fakt) --Menge aktuelle Rechnung
                       --*(IFTHEN(be_txba=11, -1, 1))    Stornorechnungen werden negativ erstellt, daher hier nicht nochmal drehen wenn Storno-Teilrechnung  --http://redmine.prodat-sql.de/issues/2802
                   )
                FROM belzeil_grund JOIN belkopf ON be_bnr=bz_be_bnr
                WHERE bz_be_bnr=r.be_bnr AND bz_add_auftg_pos IN (SELECT bz_add_auftg_pos FROM belzeil_grund WHERE bz_be_bnr=r.be_bnr) AND NOT bz_vkptotalpos;
            --
            --raise exception '%-%', r.be_bnr, n;
            s:=s+COALESCE(n,0);
        END LOOP;
    ELSE--Schlußrechnung: alle Abschlagsrechnungen werden gegengerechnet, unabhängig davon ob die Positionen übereinstimmen, oder nicht.
        FOR r IN SELECT DISTINCT be_bnr FROM belkopf JOIN belzeil_grund ON bz_be_bnr=be_bnr WHERE be_prof IN ('T') AND bz_auftg IN (SELECT bz_auftg FROM belzeil_grund WHERE bz_be_bnr=bebnr) LOOP
            s:=s+IFTHEN(brutto, IFTHEN(basis_w, be_gesamt_steu_basis_w, be_gesamt_steu), IFTHEN(basis_w, be_gesamt_net_basis_w, be_gesamt_net)) FROM belkopf WHERE be_bnr=r.be_bnr;
        END LOOP;
    END IF;
    --
    result:=s*(IFTHEN(txba=11, -1, 1));--storno rechnung, abschläge rückwärts rechnen
    --caching schreiben
    PERFORM tcache.function_cache_setcache('bel_abschlagrechbetrag', bebnr::VARCHAR, txba::VARCHAR, brutto::VARCHAR, basis_w::VARCHAR, NULL::VARCHAR, NULL::VARCHAR, NULL::VARCHAR, result);
    --end caching schreiben
    RETURN result;
  END $$ LANGUAGE plpgsql STABLE;
--
*/

 CREATE OR REPLACE FUNCTION bel_abschlagrechbetrag_basis_w(bebnr VARCHAR, brutto BOOL) RETURNS NUMERIC(16,4) AS $$
 BEGIN
  RETURN public.bel_abschlagrechbetrag(bebnr, brutto, true);
 END $$ LANGUAGE plpgsql STABLE;



 CREATE TABLE wendatgutschrift
  (wg_id                SERIAL PRIMARY KEY,
   wg_w_wen             INTEGER NOT NULL REFERENCES wendat ON UPDATE CASCADE ON DELETE CASCADE,
   wg_be_bnr            VARCHAR(15) REFERENCES belkopf ON UPDATE CASCADE,
   wg_gutstk            NUMERIC(12,4)
  );



/**************************************************************************************************************************************************************
************************ TFaktura-Funktionen. Beim Aufräumen mit 01 Func Faktura zusammenführen und NACH Tabellenerstellung verschieben ***********************
***************************************************************************************************************************************************************/

-- Prüft ob die wichtigsten Vorbedingungen erfüllt sind, damit die Rechnung in die Buchhaltung kann. ACHTUNG Überladen!
CREATE OR REPLACE FUNCTION TFaktura.ExportReady(
    IN  _dok belkopf,
    IN  _export_type varchar = null,
    OUT IsValid boolean,
    OUT ErrorString text
    )
    RETURNS     record
    AS $$
    DECLARE r   record;
    BEGIN

      IsValid := TRUE;
      _export_type := coalesce(_export_type, '');

      IF NOT _dok.be_def THEN
        IsValid := False;
        ErrorString := TSystem.FormatLines(ErrorString, lang_text(13625)); -- Nicht definitiv
      END IF;

      IF NOT EXISTS(SELECT a1_knr
                      FROM adressen_view
                      JOIN adk1 ON a1_krz = adk_ad_krz  -- Debitorendaten zur Adresse
                     WHERE ad_krz = _dok.be_rkrz AND a1_knr > 0)
      THEN
        IsValid := false;
        ErrorString := TSystem.FormatLines(ErrorString, lang_text(13630)); -- Keine Debitorendaten gefunden
      END IF;

      IF NOT TDMS.Dokument__exists__by__dbrid(_dok.dbrid)
      THEN
        IsValid := false;
        ErrorString := TSystem.FormatLines(ErrorString, lang_text('30228')); -- Kein Dokument erstellt
      END IF;

      -- Es sind DMS Einträge da, welche noch unvollständig sind, d.h. das File ist noch nicht korrekt abgelegt im DMS
      IF  EXISTS(SELECT *
                   FROM TDMS.Dokument__picndoku_records__by__dbrid(_dok.dbrid)
                  WHERE (   pd_dms_delayed_upload IS true -- warten bis APPS einliest
                         OR pd_dmsremotefile IS NULL      -- noch nicht im DMS abgelegt
                        )
                )
      THEN
        IsValid := false;
        ErrorString := TSystem.FormatLines(ErrorString, lang_text('30229')); -- Noch kein Dokument vorhanden
      END IF;

      --Kontierung und Steuercode prüfen
      FOR r IN SELECT bz_pos                        AS sortID,
                      bz_zeko                       AS konto,
                      bz_steucode                   AS steucode,
                      lang_text(15247) || bz_pos    AS ident
                 FROM belzeil_grund
                WHERE bz_be_bnr = _dok.be_bnr
                UNION
               SELECT 100002, beaz_konto, beaz_steu, abz_txt
                 FROM belabzu
                 JOIN abzu ON beaz_abz_id = abz_id
                WHERE beaz_bebnr = _dok.be_bnr
                ORDER BY sortID, ident
      LOOP

         IF     r.konto IS null
            AND TSystem.Settings__GetBool('Export_Fak_Kontierung__group_by_konto', true) -- #21960
         THEN
           IsValid := false;
           ErrorString := TSystem.FormatLines(ErrorString, r.Ident ||': ' || lang_text(13626)); -- Kontierung fehlt
         END IF;

         IF r.steucode IS null THEN
           IsValid := false;
           ErrorString := TSystem.FormatLines(ErrorString, r.Ident ||': ' || lang_text(13627)); -- Steuercode fehlt
         END IF;

      END LOOP;
      --

      RETURN;

    END $$ LANGUAGE plpgsql STABLE;
--

-- Prüft ob die wichtigsten Vorbedingungen erfüllt sind, damit die Rechnung in die Buchhaltung kann. ACHTUNG Überladen!
CREATE OR REPLACE FUNCTION TFaktura.ExportReady(
    IN  _bebnr VARCHAR,
    IN  _export_type varchar = null, -- wird aktuell auch intern nicht verwendet
    OUT IsValid boolean,
    OUT ErrorString text
    )
    RETURNS     record
    AS $$

     SELECT TFaktura.ExportReady(belkopf, _export_type) FROM belkopf
       WHERE _bebnr = be_bnr;

    $$ LANGUAGE sql STABLE;

-- keine leeren Statements am Ende vom Erstellen der DB erlaubt.
SELECT TRUE;
